#ifndef AVIFILE_AUDIOQUEUE_H
#define AVIFILE_AUDIOQUEUE_H

#include "formats.h"
#include "avm_locker.h"
#include "avm_stl.h"

AVM_BEGIN_NAMESPACE;

class IAudioResampler;
class IAudioCleaner;

// buggy gcc-4.0 refused to compile static const double
#define MAX_BUFFER_TIME 1.0

class IAudioMix
{
public:
    virtual ~IAudioMix() {}
    virtual ssize_t Mix(void* data, const void* src, size_t n) const = 0;
};

class AudioQueue
{
public:
    //static const double MAX_BUFFER_TIME = 1.0;

    AudioQueue(WAVEFORMATEX& Iwf, WAVEFORMATEX& Owf);
    ~AudioQueue();

    void Broadcast() { m_Cond.Broadcast(); }
    void Clear();
    void DisableCleaner();

    double GetBufferTime() const { return GetSize() / (double) GetBytesPerSec(); }
    size_t GetBytesPerSec() const { return m_uiBytesPerSec; }
    size_t GetSize() const { return m_uiBufSize; }

    bool IsFull() { return m_Bufs.full() || GetBufferTime() > MAX_BUFFER_TIME; }

    ssize_t Read(void* data, size_t count, const IAudioMix* amix = 0);
    ssize_t Resample(void* dest, const void* src, size_t src_size);
    ssize_t Write(char* b, size_t count); // add new samples

    int Wait(float waitTime = -1.0)
    {
        Broadcast();
	return m_Cond.Wait(m_Mutex, waitTime);
    }
    int Lock() { return m_Mutex.Lock(); }
    int Unlock() { return m_Mutex.Unlock(); }
protected:
    WAVEFORMATEX m_Iwf;		///< format of data we get from audio stream
    WAVEFORMATEX m_Owf;		///< format of data audio output

    PthreadCond m_Cond;      	///< broadcast to notify audio_thread about
				///< new available data OR about quit=1
    PthreadMutex m_Mutex;	///< controls access to audio buffer
    size_t m_uiBytesPerSec;	///< input and output positions in queue
    IAudioResampler* m_pResampler;
    double m_dRemains;          ///< remainder for resampling

    size_t m_uiBufSize;

    struct chunk {
	char* mem;              ///< memchunk
	size_t size;		///< chunk size
	size_t rsize;		///< already readed samples from this chunk
    };

    qring<chunk> m_Bufs;

    IAudioCleaner* m_pCleaner;
    bool m_bCleared;            ///< cleared buffers
};

AVM_END_NAMESPACE;

#endif // AVIFILE_AUDIOQUEUE_H
