/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the XMPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * XMPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: xmpi_asc_parse.cc,v 1.9 2003/08/23 13:46:45 jsquyres Exp $
 *
 *	Function:	- parse an application schema
 */

#include <string>

using namespace std;

#include <errno.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <all_list.h>
#include <all_opt.h>
#include <sfh.h>
#include <kio.h>

#include "app_schema.h"
#include "lam.h"

#include "xmpi.h"
#include "xmpi_asc_parse.h"
#include "xmpi_error.h"

#define LAM_PTY_IS_DEFAULT 1

static void xmpi_asc_build_rtf(OPT*, int4*);

int
xmpi_asc_parse(const char *appfile, int4 *rtf, LIST **ret_appd)
{

  struct stat fileinfo;		       /* file information */
  LIST *appd;			       /* app. schema desc. */
  int lineno = 0;		       /* line number */
  char ebuf[1024];		       /* formatting buffer */
  int bufsize;
  int fd;
  char *filebuf;
  char **env = NULL;
  int nbytes;
  int plineno;
  OPT *ad;

  /*
   * Check that the application schema is an existing regular file.
   */
  if (stat(appfile, &fileinfo)) {
    sprintf(ebuf, "Application schema \"%s\"\n", appfile);
    xmpi_error(0, ebuf);
    return (0);
  }
  if (!S_ISREG(fileinfo.st_mode)) {
    sprintf(ebuf,
	    "Application schema \"%s\"\n: Is not a regular file",
	    appfile);
    errno = 0;
    xmpi_error(0, ebuf);
    return (0);
  }

  /*
   * Get file info.
   */
  if (stat(appfile, &fileinfo)) return(0);
  
  bufsize = (int) fileinfo.st_size;
  if (bufsize <= 0) return(0);

  /*
   * Allocate a file buffer.
   */
  filebuf = (char*) malloc((unsigned) bufsize);
  if (filebuf == 0) return(0);

  /*
   * Open and read the file.
   */
  fd = open(appfile, O_RDONLY, 0);
  if (fd < 0) {
    free(filebuf);
    return 0;
  }

  nbytes = read(fd, filebuf, bufsize);
  close(fd);
  
  if (nbytes < bufsize) {
    if (nbytes >= 0) errno = EIO;
    free(filebuf);
    return 0;
  }

  //
  // Get the mpirun line from the buffer.
  //
  // dooh...  filebuf might not end in \0.  Needed for creating string
  char* tmp_string;
  tmp_string = (char*) malloc(sizeof(char) * (bufsize + 1));
  (void) memcpy(tmp_string, filebuf, bufsize);
  tmp_string[bufsize] = '\0';

  string args(tmp_string);
  free(tmp_string);
  if (args.find("# mpirun") == string::npos) {
    // make it empty...
    args.erase();
  } else {
    args.erase(0, args.find("# mpirun") + 8);
    args.erase(args.find("\n"), args.size());
  }

  // If our app schema doesn't have info on how to run this
  // application, we can get away with using the defaults just about
  // everywhere.  However, we are going to explicitly turn on tracing,
  // as that isn't a default - and making it default would be kinda
  // complicated - there is no flag to mpirun to explicitly disable
  // tracing.  -toff turns it off, but not disables it.  So, we add
  // -ton, so that tracing can occur.
  if (args.length() == 0) {
    args = "-ton";
  }

  if ((ad = xmpi_get_options((char*) args.c_str())) == 0) {
    free(filebuf);
    return 0;
  }

  // add then environment, if needed...
  if (asc_environment(!ao_taken(ad, (char*) "nx"),
		      ao_taken(ad, (char*) "x") ? 
		      ao_param(ad, (char*) "x", 0, 0) : 0, &env)) {
    free(filebuf);
    ao_free(ad);
    return 0;
  }

  // build the rtf bitmap
  xmpi_asc_build_rtf(ad, rtf);

  /*
   * Parse the app. schema buffer. (for everything but the mpirun line)
   */
  appd = asc_bufparse(filebuf, bufsize, &plineno, env);
  free(filebuf);
  
  if (appd == 0) {
    if (lineno > 0) {
      sprintf(ebuf,
	      "Parsing application schema \"%s\"\nline %d",
	      appfile, lineno);
    } else {
      sprintf(ebuf, "Parsing application schema \"%s\"\n",
	      appfile);
    }
    
    xmpi_error(0, ebuf);
    ao_free(ad);
    return (0);
  }

  // assign variables and stuff.
  ao_free(ad);
  *ret_appd = appd;
  return (1);
}

/*
 *	xmpi_asc_free
 *
 *	Function:	- free an application schema descriptor
 *	Accepts:	- app. schema desc.
 */
void
xmpi_asc_free(LIST *appd)
{
  asc_free(appd);
}


static void
xmpi_asc_build_rtf(OPT* ad, int4* rtf)
{
  *rtf = RTF_MPIRUN;

  if (!ao_taken(ad, (char*) "nsigs"))
    *rtf |= RTF_MPISIGS;
  if (ao_taken(ad, (char*) "D"))
    *rtf |= RTF_APPWD;
  if (ao_taken(ad, (char*) "O"))
    *rtf |= RTF_HOMOG;
  if (!ao_taken(ad, (char*) "lamd"))
    *rtf |= RTF_MPIC2C;
  if (ao_taken(ad, (char*) "ger")
      || (ao_taken(ad, (char*) "lamd") && !ao_taken(ad, (char*) "nger")))
    *rtf |= RTF_MPIGER;
  if (ao_taken(ad, (char*) "t") || ao_taken(ad, (char*) "ton") || 
      ao_taken(ad, (char*) "toff"))
    *rtf |= RTF_TRACE;
  if (!ao_taken(ad, (char*) "toff"))
    *rtf |= RTF_TRSWITCH;
  if (isatty(1))
    *rtf |= RTF_TTYOUT;
  if (!ao_taken(ad, (char*) "f"))
    *rtf |= RTF_IO;
#if LAM_HAVE_PTYS
#if LAM_PTY_IS_DEFAULT
  if (!ao_taken(ad, (char*) "npty"))
    *rtf |= RTF_PTYS;
#else
  if (ao_taken(ad, (char*) "pty"))
    *rtf |= RTF_PTYS;
#endif
#else
  /* For those systems that do not have pty support */
  if (ao_taken(ad, (char*) "pty"))
    fprintf(stderr, "Your system does not appear to include pty support.  -pty ignored.\n");
  *rtf &= ~RTF_PTYS;
#endif
}

OPT*
xmpi_get_options(const char* run_options)
{
  char* cmdline;
  char** cmdv;
  int cmdc;
  OPT		*ad;			/* argument descriptor */

  /*
   * Allocate one extra char for terminator and 2 extra so we can put a fake
   * argv[0] at the start and then use the usual parsing routines.
   */
  cmdline = (char*) malloc(sizeof(char) * (strlen(run_options) + 3));
  if (cmdline == 0) 
    return (OPT*) 0;

  /*
   * Insert the fake argv[0] and then copy the line.
   */
  strcpy(cmdline,(char*) "x ");
  strcat(cmdline, run_options);

  /*
   * Split it into a command line argv.
   */
  cmdv = sfh_argv_break_quoted(cmdline, ' ', (char*) "\\'\"");
  free(cmdline);
  
  if (cmdv == 0)
    return 0;
  
  cmdc = sfh_argv_count(cmdv);

  ad = ao_init();
  if (ad == 0) {
    sfh_argv_free(cmdv);
    return 0;
  }

  ao_setopt1(ad,(char*) "fhtvDO", 0, 0, 0);
  ao_setopt(ad,(char*) "client", 0, 2, 0);
  ao_setopt(ad,(char*) "server", 0, 1, 0);
  ao_setopt(ad,(char*) "s", 0, 1, 0);
  ao_setopt(ad,(char*) "c", 0, 1, AOINT);
  ao_setopt(ad,(char*) "np",(char*) "c", 1, AOINT);
  ao_setopt(ad,(char*) "c2c", 0, 0, 0);
  ao_setopt(ad,(char*) "lamd",(char*) "c2c", 0, 0);
  ao_setopt(ad,(char*) "ger", 0, 0, 0);
  ao_setopt(ad,(char*) "nger",(char*) "ger", 0, 0);
  ao_setopt(ad,(char*) "w", 0, 0, 0);
  ao_setopt(ad,(char*) "nw",(char*) "w", 0, 0);
  ao_setopt(ad,(char*) "toff", 0, 0, 0);
  ao_setopt(ad,(char*) "ton",(char*) "toff", 0, 0);
  ao_setopt(ad,(char*) "nsigs", 0, 0, 0);
  ao_setopt(ad,(char*) "x", 0, 1, 0);
  ao_setopt(ad,(char*) "nx", 0, 0, 0);
  ao_setopt(ad,(char*) "wd",(char*) "D", 1, 0);
  ao_setopt(ad,(char*) "pty", 0, 0, 0);
  ao_setopt(ad,(char*) "npty",(char*) "pty", 0, 0);

  if (asc_compat(&cmdc, &cmdv, ad)) {
    sfh_argv_free(cmdv);
    return 0;
  }
  
  if (ao_parse(ad, &cmdc, cmdv)) {
    sfh_argv_free(cmdv);
    ao_free(ad);
    return 0;
  }

  // I think we can do this.  Time to find out...
  sfh_argv_free(cmdv);
  
  return ad;
}
