// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <memory>

#include "base/task/current_thread.h"
#include "build/build_config.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/net/system_network_context_manager.h"
#include "chrome/browser/policy/chrome_browser_policy_connector.h"
#include "chrome/browser/ssl/ssl_browsertest_util.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/common/pref_names.h"
#include "chrome/test/base/chrome_test_utils.h"
#include "chrome/test/base/in_process_browser_test.h"
#include "chrome/test/base/ui_test_utils.h"
#include "components/policy/core/common/mock_configuration_policy_provider.h"
#include "components/policy/policy_constants.h"
#include "components/prefs/pref_service.h"
#include "components/security_interstitials/content/ssl_error_handler.h"
#include "content/public/browser/navigation_entry.h"
#include "content/public/browser/network_service_instance.h"
#include "content/public/browser/ssl_status.h"
#include "content/public/common/network_service_util.h"
#include "content/public/test/browser_test.h"
#include "net/cert/ev_root_ca_metadata.h"
#include "net/test/cert_test_util.h"
#include "net/test/embedded_test_server/embedded_test_server.h"
#include "services/network/public/cpp/features.h"
#include "services/network/public/mojom/ssl_config.mojom.h"
#include "third_party/blink/public/common/features.h"

namespace AuthState = ssl_test_util::AuthState;

namespace {

// SHA256 hash of the testserver root_ca_cert DER.
// openssl x509 -in root_ca_cert.pem -outform der | \
//   openssl dgst -sha256 -binary | xxd -i
static const net::SHA256HashValue kTestRootCertHash = {
    {0xb2, 0xab, 0xa3, 0xa5, 0xd4, 0x11, 0x56, 0xcb, 0xb9, 0x23, 0x35,
     0x07, 0x6d, 0x0b, 0x51, 0xbe, 0xd3, 0xee, 0x2e, 0xab, 0xe7, 0xab,
     0x6b, 0xad, 0xcc, 0x2a, 0xfa, 0x35, 0xfb, 0x8e, 0x31, 0x5e}};

// The test EV policy OID used for generated certs.
static const char kOCSPTestCertPolicy[] = "1.3.6.1.4.1.11129.2.4.1";

}  // namespace

class OCSPBrowserTest : public PlatformBrowserTest,
                        public ::testing::WithParamInterface<bool>,
                        public network::mojom::SSLConfigClient {
 public:
  OCSPBrowserTest() = default;

  void SetUp() override {
    SystemNetworkContextManager::SetEnableCertificateTransparencyForTesting(
        false);

    ON_CALL(policy_provider_, IsInitializationComplete(testing::_))
        .WillByDefault(testing::Return(true));
    ON_CALL(policy_provider_, IsFirstPolicyLoadComplete(testing::_))
        .WillByDefault(testing::Return(true));
    policy::BrowserPolicyConnector::SetPolicyProviderForTesting(
        &policy_provider_);

    InProcessBrowserTest::SetUp();
    SSLErrorHandler::ResetConfigForTesting();
  }

  void TearDown() override {
    InProcessBrowserTest::TearDown();

    SystemNetworkContextManager::SetEnableCertificateTransparencyForTesting(
        base::nullopt);
  }

  void SetUpInProcessBrowserTestFixture() override {
    std::vector<base::Feature> enabled_features;
    std::vector<base::Feature> disabled_features;
    disabled_features.push_back(blink::features::kMixedContentAutoupgrade);
    if (GetParam()) {
      enabled_features.push_back(network::features::kCertVerifierService);
    } else {
      disabled_features.push_back(network::features::kCertVerifierService);
    }

    scoped_feature_list_.InitWithFeatures(enabled_features, disabled_features);
  }

  void SetUpOnMainThread() override {
    network::mojom::NetworkContextParamsPtr context_params =
        g_browser_process->system_network_context_manager()
            ->CreateDefaultNetworkContextParams();
    last_ssl_config_ = *context_params->initial_ssl_config;
    receiver_.Bind(std::move(context_params->ssl_config_client_receiver));

    if (GetParam() || content::IsInProcessNetworkService()) {
      // TODO(https://crbug.com/1085233): when the CertVerifierService is moved
      // out of process, the ScopedTestEVPolicy needs to be instantiated in
      // that process.
      ev_test_policy_ = std::make_unique<net::ScopedTestEVPolicy>(
          net::EVRootCAMetadata::GetInstance(), kTestRootCertHash,
          kOCSPTestCertPolicy);
    } else {
      content::GetNetworkService()->BindTestInterface(
          network_service_test_.BindNewPipeAndPassReceiver());
      mojo::ScopedAllowSyncCallForTesting allow_sync_call;
      EXPECT_TRUE(network_service_test_->SetEVPolicy(
          std::vector<uint8_t>(
              kTestRootCertHash.data,
              kTestRootCertHash.data + sizeof(kTestRootCertHash.data)),
          kOCSPTestCertPolicy));
    }
  }

  // Sets the policy identified by |policy_name| to be true, ensuring
  // that the corresponding boolean pref |pref_name| is updated to match.
  void EnablePolicy(PrefService* pref_service,
                    const char* policy_name,
                    const char* pref_name) {
    policy::PolicyMap policy_map;
    policy_map.Set(policy_name, policy::POLICY_LEVEL_MANDATORY,
                   policy::POLICY_SCOPE_MACHINE, policy::POLICY_SOURCE_CLOUD,
                   base::Value(true), nullptr);

    EXPECT_NO_FATAL_FAILURE(UpdateChromePolicy(policy_map));

    EXPECT_TRUE(pref_service->GetBoolean(pref_name));
    EXPECT_TRUE(pref_service->IsManagedPreference(pref_name));

    // Wait for the updated SSL configuration to be sent to the network service,
    // to avoid a race.
    g_browser_process->system_network_context_manager()
        ->FlushSSLConfigManagerForTesting();
  }

  void EnableRevocationChecking() {
    // OCSP checking is disabled by default.
    EXPECT_FALSE(last_ssl_config_.rev_checking_enabled);
    EXPECT_FALSE(g_browser_process->system_network_context_manager()
                     ->CreateDefaultNetworkContextParams()
                     ->initial_ssl_config->rev_checking_enabled);

    // Enable, and make sure the default network context params reflect the
    // change.
    base::RunLoop run_loop;
    set_ssl_config_updated_callback(run_loop.QuitClosure());
    ASSERT_NO_FATAL_FAILURE(
        EnablePolicy(g_browser_process->local_state(),
                     policy::key::kEnableOnlineRevocationChecks,
                     prefs::kCertRevocationCheckingEnabled));
    run_loop.Run();
    EXPECT_TRUE(last_ssl_config_.rev_checking_enabled);
    EXPECT_TRUE(g_browser_process->system_network_context_manager()
                    ->CreateDefaultNetworkContextParams()
                    ->initial_ssl_config->rev_checking_enabled);
  }

  void DoConnection(
      const net::EmbeddedTestServer::ServerCertificateConfig& config) {
    net::EmbeddedTestServer server(net::EmbeddedTestServer::TYPE_HTTPS);

    server.SetSSLConfig(config);
    server.AddDefaultHandlers(GetChromeTestDataDir());
    ASSERT_TRUE(server.Start());

    ui_test_utils::NavigateToURL(browser(), server.GetURL("/ssl/google.html"));
  }

  net::CertStatus GetCurrentCertStatus() {
    content::NavigationEntry* entry =
        chrome_test_utils::GetActiveWebContents(this)
            ->GetController()
            .GetVisibleEntry();
    return entry->GetSSL().cert_status;
  }

  void set_ssl_config_updated_callback(
      const base::RepeatingClosure& ssl_config_updated_callback) {
    ssl_config_updated_callback_ = std::move(ssl_config_updated_callback);
  }

  // network::mojom::SSLConfigClient implementation.
  void OnSSLConfigUpdated(network::mojom::SSLConfigPtr ssl_config) override {
    last_ssl_config_ = *ssl_config;
    if (ssl_config_updated_callback_)
      ssl_config_updated_callback_.Run();
  }

  const network::mojom::SSLConfig& last_ssl_config() const {
    return last_ssl_config_;
  }

 private:
  void UpdateChromePolicy(const policy::PolicyMap& policies) {
    policy_provider_.UpdateChromePolicy(policies);
    ASSERT_TRUE(base::CurrentThread::Get());

    base::RunLoop().RunUntilIdle();

    content::FlushNetworkServiceInstanceForTesting();
  }

  testing::NiceMock<policy::MockConfigurationPolicyProvider> policy_provider_;

  std::unique_ptr<net::ScopedTestEVPolicy> ev_test_policy_;
  base::test::ScopedFeatureList scoped_feature_list_;
  mojo::Remote<network::mojom::NetworkServiceTest> network_service_test_;

  base::RepeatingClosure ssl_config_updated_callback_;
  network::mojom::SSLConfig last_ssl_config_;
  mojo::Receiver<network::mojom::SSLConfigClient> receiver_{this};
};

// Visits a page with revocation checking set to the default value (disabled)
// and a revoked OCSP response.
IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, TestHTTPSOCSPRevokedButNotChecked) {
  // OCSP checking is disabled by default.
  EXPECT_FALSE(last_ssl_config().rev_checking_enabled);
  EXPECT_FALSE(g_browser_process->system_network_context_manager()
                   ->CreateDefaultNetworkContextParams()
                   ->initial_ssl_config->rev_checking_enabled);

  net::EmbeddedTestServer::ServerCertificateConfig revoked_cert_config;
  revoked_cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::REVOKED,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});
  DoConnection(revoked_cert_config);

  ssl_test_util::CheckAuthenticatedState(
      chrome_test_utils::GetActiveWebContents(this), AuthState::NONE);

  net::CertStatus cert_status = GetCurrentCertStatus();
  EXPECT_FALSE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

// Visits a page with revocation checking enabled and a valid OCSP response.
IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, TestHTTPSOCSPOk) {
  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig ok_cert_config;
  ok_cert_config.policy_oids = {kOCSPTestCertPolicy};
  ok_cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});

  DoConnection(ok_cert_config);

  ssl_test_util::CheckAuthenticatedState(
      chrome_test_utils::GetActiveWebContents(this), AuthState::NONE);

  net::CertStatus cert_status = GetCurrentCertStatus();
  EXPECT_EQ(ssl_test_util::SystemUsesChromiumEVMetadata(),
            static_cast<bool>(cert_status & net::CERT_STATUS_IS_EV));
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

// Visits a page with revocation checking enabled and a revoked OCSP response.
IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, TestHTTPSOCSPRevoked) {
  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig revoked_cert_config;
  revoked_cert_config.policy_oids = {kOCSPTestCertPolicy};
  revoked_cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::REVOKED,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});

  DoConnection(revoked_cert_config);

  ssl_test_util::CheckAuthenticationBrokenState(
      chrome_test_utils::GetActiveWebContents(this), net::CERT_STATUS_REVOKED,
      AuthState::SHOWING_INTERSTITIAL);

  net::CertStatus cert_status = GetCurrentCertStatus();
  EXPECT_FALSE(cert_status & net::CERT_STATUS_IS_EV);
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, TestHTTPSOCSPInvalid) {
  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig invalid_cert_config;
  invalid_cert_config.policy_oids = {kOCSPTestCertPolicy};
  invalid_cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      net::EmbeddedTestServer::OCSPConfig::ResponseType::kInvalidResponse);

  DoConnection(invalid_cert_config);

  ssl_test_util::CheckAuthenticatedState(
      chrome_test_utils::GetActiveWebContents(this), AuthState::NONE);

  net::CertStatus cert_status = GetCurrentCertStatus();
  EXPECT_FALSE(cert_status & net::CERT_STATUS_IS_EV);
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, TestHTTPSOCSPIntermediateValid) {
  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig
      intermediate_invalid_cert_config;
  intermediate_invalid_cert_config.policy_oids = {kOCSPTestCertPolicy};
  intermediate_invalid_cert_config.intermediate =
      net::EmbeddedTestServer::IntermediateType::kInHandshake;
  intermediate_invalid_cert_config
      .ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});
  intermediate_invalid_cert_config
      .intermediate_ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});

  DoConnection(intermediate_invalid_cert_config);

  ssl_test_util::CheckAuthenticatedState(
      chrome_test_utils::GetActiveWebContents(this), AuthState::NONE);

  net::CertStatus cert_status = GetCurrentCertStatus();
  EXPECT_EQ(ssl_test_util::SystemUsesChromiumEVMetadata(),
            static_cast<bool>(cert_status & net::CERT_STATUS_IS_EV));
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

IN_PROC_BROWSER_TEST_P(OCSPBrowserTest,
                       TestHTTPSOCSPIntermediateResponseOldButStillValid) {
  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig cert_config;
  cert_config.policy_oids = {kOCSPTestCertPolicy};
  cert_config.intermediate =
      net::EmbeddedTestServer::IntermediateType::kInHandshake;
  cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});
  // Use an OCSP response for the intermediate that would be too old for a leaf
  // cert, but is still valid for an intermediate.
  cert_config.intermediate_ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kLong}});

  DoConnection(cert_config);

  ssl_test_util::CheckAuthenticatedState(
      chrome_test_utils::GetActiveWebContents(this), AuthState::NONE);

  net::CertStatus cert_status = GetCurrentCertStatus();
  EXPECT_EQ(ssl_test_util::SystemUsesChromiumEVMetadata(),
            static_cast<bool>(cert_status & net::CERT_STATUS_IS_EV));
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

IN_PROC_BROWSER_TEST_P(OCSPBrowserTest,
                       TestHTTPSOCSPIntermediateResponseTooOld) {
  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig cert_config;
  cert_config.policy_oids = {kOCSPTestCertPolicy};
  cert_config.intermediate =
      net::EmbeddedTestServer::IntermediateType::kInHandshake;
  cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});
  cert_config.intermediate_ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kLonger}});

  DoConnection(cert_config);

  ssl_test_util::CheckAuthenticatedState(
      chrome_test_utils::GetActiveWebContents(this), AuthState::NONE);

  net::CertStatus cert_status = GetCurrentCertStatus();
  if (ssl_test_util::UsingBuiltinCertVerifier()) {
    // The builtin verifier enforces the baseline requirements for max age of an
    // intermediate's OCSP response, so the connection is considered non-EV.
    EXPECT_EQ(0u, cert_status & net::CERT_STATUS_ALL_ERRORS);
    EXPECT_EQ(0u, cert_status & net::CERT_STATUS_IS_EV);
  } else {
    // The platform verifiers are more lenient.
    EXPECT_EQ(0u, cert_status & net::CERT_STATUS_ALL_ERRORS);
    EXPECT_EQ(ssl_test_util::SystemUsesChromiumEVMetadata(),
              static_cast<bool>(cert_status & net::CERT_STATUS_IS_EV));
  }
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, TestHTTPSOCSPIntermediateRevoked) {
  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig cert_config;
  cert_config.policy_oids = {kOCSPTestCertPolicy};
  cert_config.intermediate =
      net::EmbeddedTestServer::IntermediateType::kInHandshake;
  cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});
  cert_config.intermediate_ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::REVOKED,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});

  DoConnection(cert_config);

  net::CertStatus cert_status = GetCurrentCertStatus();

#if defined(OS_WIN)
  // TODO(mattm): Seems to be flaky on Windows. Either returns
  // CERT_STATUS_UNABLE_TO_CHECK_REVOCATION (which gets masked off due to
  // soft-fail), or CERT_STATUS_REVOKED.
  EXPECT_THAT(cert_status & net::CERT_STATUS_ALL_ERRORS,
              ::testing::AnyOf(0u, net::CERT_STATUS_REVOKED));
#else
  ssl_test_util::CheckAuthenticationBrokenState(
      chrome_test_utils::GetActiveWebContents(this), net::CERT_STATUS_REVOKED,
      AuthState::SHOWING_INTERSTITIAL);
#endif

  EXPECT_EQ(0u, cert_status & net::CERT_STATUS_IS_EV);
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, TestHTTPSOCSPValidStapled) {
  if (!ssl_test_util::SystemSupportsOCSPStapling()) {
    LOG(WARNING)
        << "Skipping test because system doesn't support OCSP stapling";
    return;
  }

  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig cert_config;
  cert_config.policy_oids = {kOCSPTestCertPolicy};

  // AIA OCSP url is included, but does not return a successful ocsp response.
  cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      net::EmbeddedTestServer::OCSPConfig::ResponseType::kTryLater);

  cert_config.stapled_ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});

  DoConnection(cert_config);

  ssl_test_util::CheckAuthenticatedState(
      chrome_test_utils::GetActiveWebContents(this), AuthState::NONE);

  net::CertStatus cert_status = GetCurrentCertStatus();
  EXPECT_EQ(ssl_test_util::SystemUsesChromiumEVMetadata(),
            static_cast<bool>(cert_status & net::CERT_STATUS_IS_EV));
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, TestHTTPSOCSPRevokedStapled) {
  if (!ssl_test_util::SystemSupportsOCSPStapling()) {
    LOG(WARNING)
        << "Skipping test because system doesn't support OCSP stapling";
    return;
  }

  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig cert_config;
  cert_config.policy_oids = {kOCSPTestCertPolicy};

  // AIA OCSP url is included, but does not return a successful ocsp response.
  cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      net::EmbeddedTestServer::OCSPConfig::ResponseType::kTryLater);

  cert_config.stapled_ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::REVOKED,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});

  DoConnection(cert_config);

  ssl_test_util::CheckAuthenticationBrokenState(
      chrome_test_utils::GetActiveWebContents(this), net::CERT_STATUS_REVOKED,
      AuthState::SHOWING_INTERSTITIAL);

  net::CertStatus cert_status = GetCurrentCertStatus();
  EXPECT_FALSE(cert_status & net::CERT_STATUS_IS_EV);
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, TestHTTPSOCSPOldStapledAndInvalidAIA) {
  if (!ssl_test_util::SystemSupportsOCSPStapling()) {
    LOG(WARNING)
        << "Skipping test because system doesn't support OCSP stapling";
    return;
  }

  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig cert_config;
  cert_config.policy_oids = {kOCSPTestCertPolicy};
  // Stapled response indicates good, but is too old.
  cert_config.stapled_ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kOld}});

  // AIA OCSP url is included, but does not return a successful ocsp response.
  cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      net::EmbeddedTestServer::OCSPConfig::ResponseType::kTryLater);

  DoConnection(cert_config);
  ssl_test_util::CheckAuthenticatedState(
      chrome_test_utils::GetActiveWebContents(this), AuthState::NONE);

  net::CertStatus cert_status = GetCurrentCertStatus();
  EXPECT_FALSE(cert_status & net::CERT_STATUS_IS_EV);
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, TestHTTPSOCSPOldStapledButValidAIA) {
  if (!ssl_test_util::SystemSupportsOCSPStapling()) {
    LOG(WARNING)
        << "Skipping test because system doesn't support OCSP stapling";
    return;
  }

  EnableRevocationChecking();

  net::EmbeddedTestServer::ServerCertificateConfig cert_config;
  cert_config.policy_oids = {kOCSPTestCertPolicy};

  // Stapled response indicates good, but response is too old.
  cert_config.stapled_ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kOld}});

  // AIA OCSP url is included, and returns a successful ocsp response.
  cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      {{net::OCSPRevocationStatus::GOOD,
        net::EmbeddedTestServer::OCSPConfig::SingleResponse::Date::kValid}});

  DoConnection(cert_config);
  ssl_test_util::CheckAuthenticatedState(
      chrome_test_utils::GetActiveWebContents(this), AuthState::NONE);

  net::CertStatus cert_status = GetCurrentCertStatus();
  EXPECT_EQ(ssl_test_util::SystemUsesChromiumEVMetadata(),
            static_cast<bool>(cert_status & net::CERT_STATUS_IS_EV));
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

IN_PROC_BROWSER_TEST_P(OCSPBrowserTest, HardFailOnOCSPInvalid) {
  if (!ssl_test_util::SystemSupportsHardFailRevocationChecking()) {
    LOG(WARNING) << "Skipping test because system doesn't support hard fail "
                 << "revocation checking";
    return;
  }

  // OCSP checking is disabled by default.
  EXPECT_FALSE(last_ssl_config().rev_checking_required_local_anchors);
  EXPECT_FALSE(g_browser_process->system_network_context_manager()
                   ->CreateDefaultNetworkContextParams()
                   ->initial_ssl_config->rev_checking_required_local_anchors);

  // Enable, and make sure the default network context params reflect the
  // change.
  base::RunLoop run_loop;
  set_ssl_config_updated_callback(run_loop.QuitClosure());
  ASSERT_NO_FATAL_FAILURE(
      EnablePolicy(g_browser_process->local_state(),
                   policy::key::kRequireOnlineRevocationChecksForLocalAnchors,
                   prefs::kCertRevocationCheckingRequiredLocalAnchors));
  run_loop.Run();
  EXPECT_TRUE(last_ssl_config().rev_checking_required_local_anchors);
  EXPECT_TRUE(g_browser_process->system_network_context_manager()
                  ->CreateDefaultNetworkContextParams()
                  ->initial_ssl_config->rev_checking_required_local_anchors);

  net::EmbeddedTestServer::ServerCertificateConfig cert_config;
  cert_config.policy_oids = {kOCSPTestCertPolicy};
  cert_config.ocsp_config = net::EmbeddedTestServer::OCSPConfig(
      net::EmbeddedTestServer::OCSPConfig::ResponseType::kInvalidResponse);

  DoConnection(cert_config);

  ssl_test_util::CheckAuthenticationBrokenState(
      chrome_test_utils::GetActiveWebContents(this),
      net::CERT_STATUS_UNABLE_TO_CHECK_REVOCATION,
      AuthState::SHOWING_INTERSTITIAL);

  net::CertStatus cert_status = GetCurrentCertStatus();
  // Without a positive OCSP response, we shouldn't show the EV status.
  EXPECT_FALSE(cert_status & net::CERT_STATUS_IS_EV);
  EXPECT_TRUE(cert_status & net::CERT_STATUS_REV_CHECKING_ENABLED);
}

INSTANTIATE_TEST_SUITE_P(/* no prefix */, OCSPBrowserTest, ::testing::Bool());

using AIABrowserTest = OCSPBrowserTest;

IN_PROC_BROWSER_TEST_P(AIABrowserTest, TestHTTPSAIA) {
  net::EmbeddedTestServer::ServerCertificateConfig cert_config;
  cert_config.intermediate = net::EmbeddedTestServer::IntermediateType::kByAIA;

  DoConnection(cert_config);
  ssl_test_util::CheckAuthenticatedState(
      chrome_test_utils::GetActiveWebContents(this), AuthState::NONE);
}

INSTANTIATE_TEST_SUITE_P(/* no prefix */, AIABrowserTest, ::testing::Bool());
