/*
 *  Copyright 1994-2019 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "globals.h"
#include "context.h"
#include "brandom.h"
#include "colormaps.h"
#include "images.h"
#include "sequences.h"

#ifdef WITH_WEBCAM
#define MAX_TIMERS 4
#else
#define MAX_TIMERS 3
#endif

/* Auto-change timers and modes */
static char *delay_names[MAX_TIMERS] = {
  "colormaps",
  "images",
  "sequences"
#ifdef WITH_WEBCAM
  , "webcams"
#endif
};

static int delays[MAX_TIMERS][2] = {
  { DELAY_MIN, DELAY_MAX }, // colormaps
  { DELAY_MIN, DELAY_MAX }, // images
  { DELAY_MIN, DELAY_MAX }  // sequences
#ifdef WITH_WEBCAM
  , { DELAY_MIN, DELAY_MAX }  // webcams
#endif
};

static enum ShufflerMode random_modes[MAX_TIMERS] = {
  BS_RANDOM,  // colormaps
  BS_RANDOM,  // images
  BS_SHUFFLE  // sequences
#ifdef WITH_WEBCAM
  , BS_CYCLE  // webcams
#endif
};


#ifdef WITH_WEBCAM
#include "webcam.h"

extern int webcams;
extern char *video_base;


static webcam_t *cams[MAX_CAMS];
static pthread_t thr[MAX_CAMS];

#define MAX_TRIES 5


static void
Context_open_webcam(Context_t *ctx)
{
  int i;
  uint8_t try = 0;

  parse_options();

  for (i = 0; i < MAX_CAMS; i++)
    pthread_mutex_init(&ctx->cam_mtx[i], NULL);

  for (i = 0; (i < webcams) && (try < MAX_TRIES); i++) {
  retry:
    cams[i] = xmalloc(sizeof(webcam_t));

    cams[i]->io = IO_METHOD_MMAP;
    cams[i]->fd = -1;
    cams[i]->ctx = ctx;
    cams[i]->cam_no = i;

    if (-1 != open_device(cams[i], try)) {
      if (-1 != init_device(cams[i])) {
	enumerate_cids(cams[i]);
	list_inputs(cams[i]);
	start_capturing(cams[i]);

	pthread_create(&thr[i], NULL, loop, (void *)cams[i]);
      } else {
	fprintf(stderr, "[i] Webcam: failed to initialize device #%d\n", i);
	close_device(cams[i]);
	xfree(cams[i]);
	try++;
	goto retry;
      }
    } else {
      xfree(cams[i]);
      webcams--;
    }
  }
}
#endif


Context_t *
Context_new()
{
  int i;

  Context_t *ctx = xcalloc(1, sizeof(Context_t));

  ctx->running = 1;

  VERBOSE(printf("[+] Creating buffers... "));
  for (i = 0; i < NSCREENS; i++) {
    VERBOSE(printf("%d ", i));
    ctx->buffers[i] = Buffer8_new();
  }
#ifdef WITH_WEBCAM
  {
    int k;
    for (k = 0; k < webcams; k++) {
      for (i = 0; i < CAM_SAVE; i++)
	ctx->cam_save[k][i] = Buffer8_new();
      ctx->cam_ref[k] = Buffer8_new();
    }
  }
#endif
  ctx->rgba_buffers[ACTIVE_BUFFER] = BufferRGBA_new();
  VERBOSE(printf("\n"));

#if WITH_GL
  glGenTextures(NSCREENS, ctx->textures); // TODO: delete on exit
  glGenTextures(MAX_CAMS, ctx->cam_textures); // TODO: delete on exit
#endif

  if (images != NULL) {
    VERBOSE(printf("[+] Creating images fader (%i images)\n", images->size));
    ctx->imgf = ImageFader_new(images->size);

    VERBOSE(printf("[+] Creating images timer (%d..%d)\n", delays[BD_IMAGES][0], delays[BD_IMAGES][1]));
    ctx->a_images = Alarm_new(delays[BD_IMAGES][0], delays[BD_IMAGES][1]);
  }

  if (colormaps != NULL) {
    VERBOSE(printf("[+] Creating colormaps fader (%i colormaps)\n", colormaps->size));
    ctx->cf = CmapFader_new(colormaps->size);

    VERBOSE(printf("[+] Creating colormaps timer (%d..%d)\n", delays[BD_COLORMAPS][0], delays[BD_COLORMAPS][1]));
    ctx->a_cmaps = Alarm_new(delays[BD_COLORMAPS][0], delays[BD_COLORMAPS][1]);
  }

  VERBOSE(printf("[+] Creating sequences timer (%d..%d)\n", delays[BD_SEQUENCES][0], delays[BD_SEQUENCES][1]));
  ctx->a_random = Alarm_new(delays[BD_SEQUENCES][0], delays[BD_SEQUENCES][1]);
  ctx->random_mode = BR_NONE;

  VERBOSE(printf("[+] Initializing sequence manager\n"));
  ctx->sm = SequenceManager_new();

  Context_load_banks(ctx);

  VERBOSE(printf("[+] Initializing 3D engine\n"));
  Params3d_init(&ctx->params3d);

  ctx->events = NULL;
  ctx->frames = ctx->nb_events = 0;
  for (i = 0; i < NFPS; i++)
    ctx->fps[i] = 0;

  ctx->timer = b_timer_new();
  ctx->fps_timer = b_timer_new();

  ctx->display_colormap = 0;

  ctx->outputs = NULL;

  ctx->target_pic = Image8_new();

  extern char *data_dir;
  gchar *tmp;
  int res;

  tmp = g_strdup_printf("%s/%s", data_dir, "images/z-biniou-tv-1.png");
  VERBOSE(printf("[+] Loading '%s'\n", tmp));
  res = Image8_load_PNG(ctx->target_pic, tmp);
  g_free(tmp);
  if (res == -1) {
    Buffer8_randomize(ctx->target_pic->buff);
  }

#ifdef WITH_WEBCAM
  VERBOSE(printf("[i] Initializing %d webcams base: %s\n", webcams, video_base));

  ctx->cam = 0;
  Context_open_webcam(ctx);

  for (i = 0; i < webcams; i++)
    Buffer8_copy(ctx->target_pic->buff, ctx->cam_save[i][0]);

  if (webcams > 1) {
    VERBOSE(printf("[+] Creating webcams timer (%d..%d)\n", delays[BD_WEBCAMS][0], delays[BD_WEBCAMS][1]));
    ctx->a_webcams = Alarm_new(delays[BD_WEBCAMS][0], delays[BD_WEBCAMS][1]);
    Alarm_init(ctx->a_webcams);

    VERBOSE(printf("[+] Creating webcams shuffler (%d webcams)\n", webcams));
    ctx->webcams_shuffler = Shuffler_new(webcams);
    Shuffler_set_mode(ctx->webcams_shuffler, Context_get_shuffler_mode(BD_WEBCAMS));
#ifdef DEBUG
    Shuffler_verbose(ctx->webcams_shuffler);
#endif
  }
#endif

  return ctx;
}


#ifdef WITH_WEBCAM
static void
Context_close_webcam(const u_char cam_no)
{
  if (NULL != cams[cam_no]) {
    pthread_join(thr[cam_no], NULL);
    stop_capturing(cams[cam_no]);
    uninit_device(cams[cam_no]);
    close_device(cams[cam_no]);
  }
}
#endif


void
Context_delete(Context_t *ctx)
{
  int i;
  GSList *outputs = ctx->outputs, *_outputs = outputs;

#ifdef WITH_WEBCAM
  VERBOSE(printf("[i] Closing %d webcams\n", webcams));
  if (webcams) {
    int i;

    for (i = 0; i < webcams; i++)
      Context_close_webcam(i);
  }
  xfree(video_base);
  if (webcams > 1) {
    Shuffler_delete(ctx->webcams_shuffler);
    Alarm_delete(ctx->a_webcams);
  }
#endif

  VERBOSE(printf("[i] %lu frames, %lu events\n", ctx->frames, ctx->nb_events));

  if (ctx->input_plugin != NULL)
    Plugin_delete(ctx->input_plugin);

  for ( ; outputs != NULL; outputs = g_slist_next(outputs)) {
    Plugin_t *output = (Plugin_t *)outputs->data;

    Plugin_delete(output);
  }
  g_slist_free(_outputs);

  VERBOSE(printf("[+] Freeing buffers... "));
  for (i = 0; i < NSCREENS; i++) {
    VERBOSE(printf("%d ", i));
    Buffer8_delete(ctx->buffers[i]);
  }
#ifdef WITH_WEBCAM
  {
    int k;
    for (k = 0; k < webcams; k++) {
      for (i = 0; i < CAM_SAVE; i++)
	Buffer8_delete(ctx->cam_save[k][i]);
      Buffer8_delete(ctx->cam_ref[k]);
    }
  }
#endif
  BufferRGBA_delete(ctx->rgba_buffers[ACTIVE_BUFFER]);
  VERBOSE(printf("\n"));

  if (ctx->imgf != NULL) {
    VERBOSE(printf("[+] Freeing images fader\n"));
    ImageFader_delete(ctx->imgf);

    VERBOSE(printf("[+] Freeing images timer\n"));
    Alarm_delete(ctx->a_images);
  }

  if (ctx->cf != NULL) {
    VERBOSE(printf("[+] Freeing colormaps fader\n"));
    CmapFader_delete(ctx->cf);

    VERBOSE(printf("[+] Freeing colormaps timer\n"));
    Alarm_delete(ctx->a_cmaps);
  }

  Alarm_delete(ctx->a_random);
  SequenceManager_delete(ctx->sm);
  b_timer_delete(ctx->timer);
  b_timer_delete(ctx->fps_timer);

  if (NULL != ctx->target_pic)
    Image8_delete(ctx->target_pic);

  xfree(ctx);
}


void
Context_set_colormap(Context_t *ctx)
{
  /* find the cmap, if any. default cmap otherwise */
  if (ctx->cf != NULL) {
    ctx->cf->fader->target
      = (ctx->sm->next->cmap_id) ?
      Colormaps_index(ctx->sm->next->cmap_id) : 0;
    CmapFader_set(ctx->cf);
  }
}


void
Context_set_image(Context_t *ctx)
{
  /* find the image, if any. default image otherwise */
  if (ctx->imgf != NULL) {
    ctx->imgf->fader->target
      = (ctx->sm->next->image_id) ?
      Images_index(ctx->sm->next->image_id) : 0;
    ImageFader_set(ctx->imgf);
  }
}

void
Context_set(Context_t *ctx)
{
  GList *tmp;

  tmp = g_list_first(ctx->sm->cur->layers);

  /* call on_switch_off() on old plugins */
  while (tmp != NULL) {
    Layer_t *layer = tmp->data;
    Plugin_t *p = layer->plugin;

    assert(p != NULL);
    if (p->on_switch_off != NULL)
      if (!(*p->options & BEQ_DISABLED))
	p->on_switch_off(ctx);
    tmp = g_list_next(tmp);
  }

  Context_set_colormap(ctx);
  Context_set_image(ctx);

  /* call on_switch_on() on new plugins */
  tmp = g_list_first(ctx->sm->next->layers);
  while (tmp != NULL) {
    Layer_t *layer = (Layer_t *)tmp->data;
    Plugin_t *p = layer->plugin;

    assert(p != NULL);
    if (p->on_switch_on != NULL)
      if (!(*p->options & BEQ_DISABLED))
	p->on_switch_on(ctx);
    tmp = g_list_next(tmp);
  }

  Sequence_copy(ctx->sm->next, ctx->sm->cur);

  Context_update_auto(ctx);

  Sequence_display(ctx->sm->cur);
  okdone("Context_set");
}


void
Context_update_auto(Context_t *ctx)
{
  /* set auto stuff */
  if (ctx->imgf != NULL) {
    ctx->imgf->on = ctx->sm->cur->auto_images;
    if (ctx->imgf->on && (images != NULL) && (images->size > 1)) {
      /* select random image and reinitialize timer */
      ImageFader_random(ctx->imgf);
      Alarm_init(ctx->a_images);
    }
  }

  if (ctx->cf != NULL) {
    ctx->cf->on = ctx->sm->cur->auto_colormaps;
    if (ctx->cf->on && (colormaps->size > 1)) {
      /* select random colormap and reinitialize timer */
      CmapFader_random(ctx->cf);
      Alarm_init(ctx->a_cmaps);
    }
  }
}


int
Context_add_rand(Sequence_t *seq,
		 const enum PluginOptions options, const int not_lens)
{
  Plugin_t *p;

  do {
    p = Plugins_get_random(options);
    if (p == NULL)
      return -1;
  } while (Sequence_find(seq, p) != NULL);

  Sequence_insert(seq, p);

  if ((*p->options & BE_LENS) && !not_lens && (seq->lens == NULL))
    seq->lens = p;

  return 0;
}


void
Context_randomize(Context_t *ctx)
{
  Sequence_t *new = ctx->sm->next;
  int rand;

  /* Use random image */
  if (images != NULL) {
    if (images->size > 1) {
      rand = b_rand_int_range(0, images->size-1);
      new->auto_images = b_rand_boolean();
    } else
      rand = new->auto_images = 0;
    new->image_id = images->imgs[rand]->id;
  }

  /* Use random colormap */
  assert(colormaps != NULL);
  if (colormaps->size > 1) {
    rand = b_rand_int_range(0, colormaps->size-1);
    new->auto_colormaps = b_rand_boolean();
  } else
    rand =new->auto_colormaps = 0;
  new->cmap_id = colormaps->cmaps[rand]->id;

  /* Set 3D rotations */
  if (b_rand_boolean())
    Params3d_randomize(&ctx->params3d);
  else
    ctx->params3d.do_auto_rotate = 0;
}


void
Context_insert_plugin(Context_t *ctx, Plugin_t *p)
{
  /* switch the plugin on */
  if (p->on_switch_on != NULL) {
    VERBOSE(printf("[i] on_switch_on '%s' (%s)\n", p->name, p->dname));
    p->on_switch_on(ctx);
  }

  Sequence_insert(ctx->sm->cur, p);
}


void
Context_remove_plugin(Context_t *ctx, Plugin_t *p)
{
  /* switch the plugin off */
  if (p->on_switch_off != NULL) {
    VERBOSE(printf("[i] on_switch_off '%s' (%s)\n", p->name, p->dname));
    p->on_switch_off(ctx);
  }

  Sequence_remove(ctx->sm->cur, p);
}


void
Context_set_max_fps(Context_t *ctx, const u_short max_fps)
{
  ctx->sync_fps = 1;
  assert(max_fps);
  ctx->max_fps = max_fps;
  ctx->i_max_fps = 1.0 / ctx->max_fps;
}


void
Context_set_engine_random_mode(Context_t *ctx, const enum RandomMode r)
{
  ctx->random_mode = r;
}


float
Context_fps(const Context_t *ctx)
{
  float mfps = 0.0;
  int i;

  for (i = 0; i < NFPS; i++)
    mfps += ctx->fps[i];
  return (mfps / (float)NFPS);
}


void
Context_previous_sequence(Context_t *ctx)
{
  Sequence_t *s;

  if (ctx->sm->curseq == NULL)
    return;

  if (ctx->sm->curseq->prev != NULL)
    ctx->sm->curseq = ctx->sm->curseq->prev;
  else
    ctx->sm->curseq = g_list_last(sequences->seqs);

  s = (Sequence_t *)ctx->sm->curseq->data;
  Sequence_copy(s, ctx->sm->next);

  Context_set(ctx);
}


void
Context_next_sequence(Context_t *ctx)
{
  Sequence_t *s;

  if (ctx->sm->curseq == NULL)
    return;

  if (ctx->sm->curseq->next != NULL)
    ctx->sm->curseq = ctx->sm->curseq->next;
  else
    ctx->sm->curseq = sequences->seqs;

  s = (Sequence_t *)ctx->sm->curseq->data;
  Sequence_copy(s, ctx->sm->next);

  Context_set(ctx);
}


void
Context_latest_sequence(Context_t *ctx)
{
  Sequence_t *s;

  if (ctx->sm->curseq == NULL)
    return;

  ctx->sm->curseq = sequences->seqs;

  s = (Sequence_t *)ctx->sm->curseq->data;
  Sequence_copy(s, ctx->sm->next);

  Context_set(ctx);
}


void
Context_random_sequence(Context_t *ctx)
{
  u_short rand;
  GList *tmp;

  rand = Shuffler_get(sequences->shuffler);

  tmp = g_list_nth(sequences->seqs, rand);
  assert(tmp != NULL);

  VERBOSE(printf("[s] Random sequence: %d\n", rand));

  ctx->sm->curseq = tmp;
  Sequence_copy(tmp->data, ctx->sm->next);

  Context_set(ctx);
}


void
Context_set_sequence(Context_t *ctx, const uint32_t id)
{
  Sequence_t *seq = Sequences_find(id);

  if (NULL == seq)
    if (ctx->sm->transient->id == id)
      seq = ctx->sm->transient;

  assert(NULL != seq);
  ctx->sm->curseq = seq->layers;
  Sequence_copy(seq, ctx->sm->next);

  Context_set(ctx);
}


void
Context_use_sequence_bank(Context_t *ctx, const u_char bank)
{
  u_long id;

  id = ctx->banks[SEQUENCES][ctx->bankset[SEQUENCES]][bank];
  if (id) {
    Sequence_t *seq;

    printf("[i] Using sequence in bank #%d\n", (bank+1));

    VERBOSE(printf("[s] Set sequence: %li\n", id));

    seq = Sequences_find(id);
    if (NULL == seq)
      if (ctx->sm->transient->id == id)
	seq = ctx->sm->transient;
    assert(NULL != seq);
    ctx->sm->curseq = seq->layers;
    Sequence_copy(seq, ctx->sm->next);

    ctx->bank[SEQUENCES] = bank;

    Context_set(ctx);
  } else
    printf("[i] Bank %d/%d is empty\n", ctx->bankset[SEQUENCES]+1, bank+1);
}


void
Context_clear_bank(Context_t *ctx, const u_char bank)
{
  ctx->banks[ctx->bank_mode][ctx->bankset[ctx->bank_mode]][bank] = 0;
}


Buffer8_t *
active_buffer(const Context_t *ctx)
{
  return ctx->buffers[ACTIVE_BUFFER];
}


Buffer8_t *
passive_buffer(const Context_t *ctx)
{
  return ctx->buffers[PASSIVE_BUFFER];
}


#ifdef WITH_WEBCAM
void
Context_push_webcam(Context_t *ctx, Buffer8_t *buff, const int cam)
{
  int i;

  Buffer8_delete(ctx->cam_save[cam][CAM_SAVE-1]);
  for (i = CAM_SAVE-1; i >= 1; i--)
    ctx->cam_save[cam][i] = ctx->cam_save[cam][i-1];
  ctx->cam_save[cam][0] = buff;
}
#endif


void
Context_set_delay(const enum RandomDelays what, const int min, const int max)
{
  if ((max <= min) || (min < 1) || (max < 1)) {
    xerror("Set random delay for %s: max (%d) must be > min (%d), both values must be > 1\n",
	   delay_names[what], max, min);
  }

#ifdef DEBUG
  printf("[i] Setting min..max delays for %s: %d..%d\n", delay_names[what], min, max);
#endif
  delays[what][0] = min;
  delays[what][1] = max;
}


void
Context_set_shuffler_mode(const enum RandomDelays what, const enum ShufflerMode mode)
{
  random_modes[what] = mode;
}


enum ShufflerMode
Context_get_shuffler_mode(const enum RandomDelays what)
{
  return random_modes[what];
}


void
Context_set_input_size(Context_t *ctx, const uint32_t input_size)
{
  ctx->input_size = input_size;
}


uint32_t
Context_get_input_size(const Context_t *ctx)
{
  return ctx->input_size;
}


void
Context_set_phase_space_delay(Context_t *ctx, const uint8_t phase_space_delay)
{
  ctx->phase_space_delay = phase_space_delay;
}


uint8_t
Context_get_phase_space_delay(const Context_t *ctx)
{
  return ctx->phase_space_delay;
}


uint32_t
Context_get_phase_space_samples(const Context_t *ctx)
{
  if (ctx->input != NULL) {
    return ctx->input->size - 2 * ctx->phase_space_delay;
  } else {
    return 0;
  }
}


void
Context_set_span_size(Context_t *ctx, const uint8_t span_size)
{
  assert(span_size > 0);
  ctx->span_size = span_size;
}


uint8_t
Context_get_span_size(const Context_t *ctx)
{
  return ctx->span_size;
}
