#ifndef OSMIUM_VISITOR_HPP
#define OSMIUM_VISITOR_HPP

/*

This file is part of Osmium (https://osmcode.org/libosmium).

Copyright 2013-2019 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <osmium/fwd.hpp>
#include <osmium/io/reader_iterator.hpp> // IWYU pragma: keep
#include <osmium/memory/buffer.hpp>
#include <osmium/osm.hpp>
#include <osmium/osm/entity.hpp>
#include <osmium/osm/item_type.hpp>

#include <type_traits>
#include <utility>

namespace osmium {

    namespace detail {

        template <typename T, typename U>
        using ConstIfConst = typename std::conditional<std::is_const<T>::value, typename std::add_const<U>::type, U>::type;

        template <typename THandler, typename TItem>
        inline void apply_item_impl(TItem& item, THandler&& handler) {
            switch (item.type()) {
                case osmium::item_type::undefined:
                    break;
                case osmium::item_type::node:
                    std::forward<THandler>(handler).osm_object(static_cast<ConstIfConst<TItem, osmium::OSMObject>&>(item));
                    std::forward<THandler>(handler).node(static_cast<ConstIfConst<TItem, osmium::Node>&>(item));
                    break;
                case osmium::item_type::way:
                    std::forward<THandler>(handler).osm_object(static_cast<ConstIfConst<TItem, osmium::OSMObject>&>(item));
                    std::forward<THandler>(handler).way(static_cast<ConstIfConst<TItem, osmium::Way>&>(item));
                    break;
                case osmium::item_type::relation:
                    std::forward<THandler>(handler).osm_object(static_cast<ConstIfConst<TItem, osmium::OSMObject>&>(item));
                    std::forward<THandler>(handler).relation(static_cast<ConstIfConst<TItem, osmium::Relation>&>(item));
                    break;
                case osmium::item_type::area:
                    std::forward<THandler>(handler).osm_object(static_cast<ConstIfConst<TItem, osmium::OSMObject>&>(item));
                    std::forward<THandler>(handler).area(static_cast<ConstIfConst<TItem, osmium::Area>&>(item));
                    break;
                case osmium::item_type::changeset:
                    std::forward<THandler>(handler).changeset(static_cast<ConstIfConst<TItem, osmium::Changeset>&>(item));
                    break;
                case osmium::item_type::tag_list:
                    std::forward<THandler>(handler).tag_list(static_cast<ConstIfConst<TItem, osmium::TagList>&>(item));
                    break;
                case osmium::item_type::way_node_list:
                    std::forward<THandler>(handler).way_node_list(static_cast<ConstIfConst<TItem, osmium::WayNodeList>&>(item));
                    break;
                case osmium::item_type::relation_member_list:
                case osmium::item_type::relation_member_list_with_full_members:
                    std::forward<THandler>(handler).relation_member_list(static_cast<ConstIfConst<TItem, osmium::RelationMemberList>&>(item));
                    break;
                case osmium::item_type::outer_ring:
                    std::forward<THandler>(handler).outer_ring(static_cast<ConstIfConst<TItem, osmium::OuterRing>&>(item));
                    break;
                case osmium::item_type::inner_ring:
                    std::forward<THandler>(handler).inner_ring(static_cast<ConstIfConst<TItem, osmium::InnerRing>&>(item));
                    break;
                case osmium::item_type::changeset_discussion:
                    std::forward<THandler>(handler).changeset_discussion(static_cast<ConstIfConst<TItem, osmium::ChangesetDiscussion>&>(item));
                    break;
            }
        }

        template <typename THandler>
        inline void apply_item_impl(const osmium::OSMEntity& item, THandler&& handler) {
            switch (item.type()) {
                case osmium::item_type::node:
                    std::forward<THandler>(handler).osm_object(static_cast<const osmium::OSMObject&>(item));
                    std::forward<THandler>(handler).node(static_cast<const osmium::Node&>(item));
                    break;
                case osmium::item_type::way:
                    std::forward<THandler>(handler).osm_object(static_cast<const osmium::OSMObject&>(item));
                    std::forward<THandler>(handler).way(static_cast<const osmium::Way&>(item));
                    break;
                case osmium::item_type::relation:
                    std::forward<THandler>(handler).osm_object(static_cast<const osmium::OSMObject&>(item));
                    std::forward<THandler>(handler).relation(static_cast<const osmium::Relation&>(item));
                    break;
                case osmium::item_type::area:
                    std::forward<THandler>(handler).osm_object(static_cast<const osmium::OSMObject&>(item));
                    std::forward<THandler>(handler).area(static_cast<const osmium::Area&>(item));
                    break;
                case osmium::item_type::changeset:
                    std::forward<THandler>(handler).changeset(static_cast<const osmium::Changeset&>(item));
                    break;
                default:
                    throw osmium::unknown_type{};
            }
        }

        template <typename THandler>
        inline void apply_item_impl(osmium::OSMEntity& item, THandler&& handler) {
            switch (item.type()) {
                case osmium::item_type::node:
                    std::forward<THandler>(handler).osm_object(static_cast<osmium::OSMObject&>(item));
                    std::forward<THandler>(handler).node(static_cast<osmium::Node&>(item));
                    break;
                case osmium::item_type::way:
                    std::forward<THandler>(handler).osm_object(static_cast<osmium::OSMObject&>(item));
                    std::forward<THandler>(handler).way(static_cast<osmium::Way&>(item));
                    break;
                case osmium::item_type::relation:
                    std::forward<THandler>(handler).osm_object(static_cast<osmium::OSMObject&>(item));
                    std::forward<THandler>(handler).relation(static_cast<osmium::Relation&>(item));
                    break;
                case osmium::item_type::area:
                    std::forward<THandler>(handler).osm_object(static_cast<osmium::OSMObject&>(item));
                    std::forward<THandler>(handler).area(static_cast<osmium::Area&>(item));
                    break;
                case osmium::item_type::changeset:
                    std::forward<THandler>(handler).changeset(static_cast<osmium::Changeset&>(item));
                    break;
                default:
                    throw osmium::unknown_type{};
            }
        }

        template <typename THandler>
        inline void apply_item_impl(const osmium::OSMObject& item, THandler&& handler) {
            switch (item.type()) {
                case osmium::item_type::node:
                    std::forward<THandler>(handler).osm_object(item);
                    std::forward<THandler>(handler).node(static_cast<const osmium::Node&>(item));
                    break;
                case osmium::item_type::way:
                    std::forward<THandler>(handler).osm_object(item);
                    std::forward<THandler>(handler).way(static_cast<const osmium::Way&>(item));
                    break;
                case osmium::item_type::relation:
                    std::forward<THandler>(handler).osm_object(item);
                    std::forward<THandler>(handler).relation(static_cast<const osmium::Relation&>(item));
                    break;
                case osmium::item_type::area:
                    std::forward<THandler>(handler).osm_object(item);
                    std::forward<THandler>(handler).area(static_cast<const osmium::Area&>(item));
                    break;
                default:
                    throw osmium::unknown_type{};
            }
        }

        template <typename THandler>
        inline void apply_item_impl(osmium::OSMObject& item, THandler&& handler) {
            switch (item.type()) {
                case osmium::item_type::node:
                    std::forward<THandler>(handler).osm_object(item);
                    std::forward<THandler>(handler).node(static_cast<osmium::Node&>(item));
                    break;
                case osmium::item_type::way:
                    std::forward<THandler>(handler).osm_object(item);
                    std::forward<THandler>(handler).way(static_cast<osmium::Way&>(item));
                    break;
                case osmium::item_type::relation:
                    std::forward<THandler>(handler).osm_object(item);
                    std::forward<THandler>(handler).relation(static_cast<osmium::Relation&>(item));
                    break;
                case osmium::item_type::area:
                    std::forward<THandler>(handler).osm_object(item);
                    std::forward<THandler>(handler).area(static_cast<osmium::Area&>(item));
                    break;
                default:
                    throw osmium::unknown_type{};
            }
        }

    } // namespace detail

    template <typename TItem, typename... THandlers>
    inline void apply_item(TItem& item, THandlers&&... handlers) {
        (void)std::initializer_list<int>{
            (detail::apply_item_impl(item, std::forward<THandlers>(handlers)), 0)...
        };
    }

    template <typename... THandlers>
    inline void apply_flush(THandlers&&... handlers) {
        (void)std::initializer_list<int>{
            (std::forward<THandlers>(handlers).flush(), 0)...
        };
    }

    template <typename TIterator, typename... THandlers>
    inline void apply(TIterator it, TIterator end, THandlers&&... handlers) {
        for (; it != end; ++it) {
            apply_item(*it, std::forward<THandlers>(handlers)...);
        }
        apply_flush(std::forward<THandlers>(handlers)...);
    }

    template <typename TContainer, typename... THandlers>
    inline void apply(TContainer& c, THandlers&&... handlers) {
        using std::begin;
        using std::end;
        apply(begin(c), end(c), std::forward<THandlers>(handlers)...);
    }

    template <typename... THandlers>
    inline void apply(const osmium::memory::Buffer& buffer, THandlers&&... handlers) {
        apply(buffer.cbegin(), buffer.cend(), std::forward<THandlers>(handlers)...);
    }

} // namespace osmium

#endif // OSMIUM_VISITOR_HPP
