//
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// field_expr_v2_nonlinear_terminal: terminals of nonn-linear expressions
// 1) class-function
//    1.1) base class for the class-function family
//    1.2) general function or class-function
//    1.3) normal to a surface
//    1.4) h_local
//    1.5) penalty
// 2) field and such
//    2.1) base class for the field family
//    2.2) field
//    2.3) grad of a field
//    2.4)  div of a field
//    2.5) curl of a field
//    2.6) jump of a field
// 3) convected field, as compose(uh,X) where X is a characteristic
//
#include "rheolef/geo_domain.h"
#include "rheolef/field_expr_v2_nonlinear_terminal.h"
#include "rheolef/piola.h"
#include "rheolef/field_evaluate.h"

namespace rheolef { namespace details {

// ---------------------------------------------------------------------------
// 1.1) base class for all class-functions
// ---------------------------------------------------------------------------
template<class T>
field_expr_v2_nonlinear_terminal_function_base_rep<T>::field_expr_v2_nonlinear_terminal_function_base_rep ()
  : _omega(),
    _piola_ops(),
    _dis_inod()
{
}
template<class T>
field_expr_v2_nonlinear_terminal_function_base_rep<T>::field_expr_v2_nonlinear_terminal_function_base_rep (const field_expr_v2_nonlinear_terminal_function_base_rep<T>& x)
  : _omega(x._omega),
    _piola_ops(x._piola_ops),
    _dis_inod(x._dis_inod)
{
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_base_rep<T>::initialize (
  const geo_basic<float_type,memory_type>& omega,
  const quadrature<float_type>&            quad) const
{
  _omega = omega;
  _piola_ops.set (quad, omega.get_piola_basis());
  return true;
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_base_rep<T>::initialize (
  const space_basic<float_type,memory_type>& Xh) const
{
  _omega = Xh.get_geo();
  _piola_ops.set (Xh.get_numbering().get_basis(), _omega.get_piola_basis());
  return true;
}
template<class T>
void
field_expr_v2_nonlinear_terminal_function_base_rep<T>::evaluate_initialize (
  const geo_element& K) const
{
  _omega.dis_inod (K, _dis_inod);
}
// TODO: move as geo::get_side (K, sid) ?
template<class T, class M>
static
const geo_element& 
global_get_side (
  const geo_basic<T,M>&        omega,
  const geo_element&           L,
  const side_information_type& sid)
{
  typedef geo_element::size_type size_type;
  size_type dis_isid = std::numeric_limits<size_type>::max();
  switch (sid.dim) {
    case 0: {
      size_type dis_inod = L [sid.loc_isid];
      dis_isid = omega.dis_inod2dis_iv (dis_inod);
      break;
    }
    case 1: dis_isid = L.edge (sid.loc_isid); break;
    case 2: dis_isid = L.face (sid.loc_isid); break;
    default: error_macro ("domain: unexpected side dimension " << sid.dim);
  }
  // TODO: pas tres clair quand on est sur un geo_domain_indirect ou un geo_domain
  //       dans quel systeme de numerotation doit appartenir la face ?
  if (omega.variant() != geo_abstract_base_rep<T>::geo_domain_indirect) {
    return omega.dis_get_geo_element (sid.dim, dis_isid);
  } else {
    return omega.get_background_geo().dis_get_geo_element (sid.dim, dis_isid);
  }
}
template<class T>
const geo_element& 
field_expr_v2_nonlinear_terminal_function_base_rep<T>::get_side (
  const geo_element&           L,
  const side_information_type& sid) const
{
  return global_get_side (_omega, L, sid);
}
// ----------------------------------------------------------------------------
// 1.3) normal() class-function
// ----------------------------------------------------------------------------
template<class T>
field_expr_v2_nonlinear_terminal_function_rep<normal_pseudo_function<T> >::field_expr_v2_nonlinear_terminal_function_rep(
    const function_type&)
 : base()
{
}
template<class T>
field_expr_v2_nonlinear_terminal_function_rep<normal_pseudo_function<T> >::field_expr_v2_nonlinear_terminal_function_rep(
    const field_expr_v2_nonlinear_terminal_function_rep<normal_pseudo_function<T> >& x)
 : base(x)
{
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_rep<normal_pseudo_function<T> >::initialize (
    const geo_basic<float_type,memory_type>& omega, 
    const quadrature<float_type>&            quad) const
{
  base::initialize (omega, quad);
  if (base::_omega.get_background_geo().map_dimension() == 1) {
    // normal computation on 0D sides requires 1D bgd mesh with neighbours
    base::_omega.get_background_geo().neighbour_guard();
  }
  return true;
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_rep<normal_pseudo_function<T> >::initialize (
    const space_basic<float_type,memory_type>& Xh) const
{
  base::initialize (Xh);
  if (base::_omega.get_background_geo().map_dimension() == 1) {
    // normal computation on 0D sides requires 1D bgd mesh with neighbours
    base::_omega.get_background_geo().neighbour_guard();
  }
  return true;
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_rep<normal_pseudo_function<T> >::evaluate (
    const geo_element&        K,
    std::vector<result_type>& value) const
{
  base::evaluate_initialize (K);
  reference_element hat_K = K.variant();
  value.resize (base::_piola_ops.pointset_size(hat_K));
  size_type q = 0;
  tensor_basic<T> DF;
  for (typename std::vector<result_type>::iterator
	  iter = value.begin(),
	  last = value.end(); iter != last; ++iter, ++q) {
    jacobian_piola_transformation (base::_omega, base::_piola_ops, K, base::_dis_inod, q, DF);
    point_basic<T> nq = normal_from_piola_transformation (base::_omega, K, DF, base::_omega.dimension());
    *iter = nq;
  }
  return true;
}
template<class T>
void
field_expr_v2_nonlinear_terminal_function_rep<normal_pseudo_function<T> >::evaluate_on_side (
    const geo_element&           L,
    const side_information_type& sid,
    std::vector<result_type>&    value) const
{
  evaluate (base::get_side(L, sid), value);
}
// ----------------------------------------------------------------------------
// 1.4) h_local() class-function
// ----------------------------------------------------------------------------
template<class T>
field_expr_v2_nonlinear_terminal_function_rep<h_local_pseudo_function<T> >::field_expr_v2_nonlinear_terminal_function_rep(
    const function_type&)
 : base()
{
}
template<class T>
field_expr_v2_nonlinear_terminal_function_rep<h_local_pseudo_function<T> >::field_expr_v2_nonlinear_terminal_function_rep(
    const field_expr_v2_nonlinear_terminal_function_rep<h_local_pseudo_function<T> >& x)
 : base(x)
{
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_rep<h_local_pseudo_function<T> >::initialize (
    const geo_basic<float_type,memory_type>& omega, 
    const quadrature<float_type>&            quad) const
{
  return base::initialize (omega, quad);
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_rep<h_local_pseudo_function<T> >::initialize (
    const space_basic<float_type,memory_type>& Xh) const
{
  return base::initialize (Xh);
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_rep<h_local_pseudo_function<T> >::evaluate (
    const geo_element&        K,
    std::vector<result_type>& value) const
{
  base::evaluate_initialize (K);
  size_type d = base::_omega.dimension();
  size_type K_map_d = K.dimension();
  tensor_basic<T> DF;
  reference_element hat_K = K.variant();
  value.resize (base::_piola_ops.pointset_size(hat_K));
  if (K_map_d != 0) {
    size_type q = 0;
    for (typename std::vector<result_type>::iterator
	  iter = value.begin(),
	  last = value.end(); iter != last; ++iter, ++q) {
      jacobian_piola_transformation (base::_omega, base::_piola_ops, K, base::_dis_inod, q, DF);
      T det_DF = det_jacobian_piola_transformation (DF, d , K_map_d);
      T h_loc = pow (abs(det_DF)/measure(hat_K), 1./K_map_d);
      *iter = h_loc;
    }
  } else {
    // d == 0: an "interface" between two edges in 1D
    //   => h_local = average of two edge length
    size_type L_dis_ie[2] = {K.master(0), K.master(1)};
    size_t n = (L_dis_ie[0] == std::numeric_limits<size_type>::max()) ? 0 :
               (L_dis_ie[1] == std::numeric_limits<size_type>::max()) ? 1 : 2;
    if (n == 0) {
      // isolated point(s) in a 0D mesh: h_local=1 by default
      std::fill (value.begin(), value.end(), result_type(1));
      return true;
    }
    size_t L_map_d = 1;
    Float h_loc = 0;
    for (size_t i = 0; i < n; ++i) {
      const geo_element& Li = base::_omega.dis_get_geo_element (L_map_d, L_dis_ie[i]);
      const point& x0 = base::_omega.dis_node (Li[0]);
      const point& x1 = base::_omega.dis_node (Li[1]);
      h_loc += norm (x1-x0)/n;
    }
    std::fill (value.begin(), value.end(), result_type(h_loc));
  }
  return true;
}
template<class T>
void
field_expr_v2_nonlinear_terminal_function_rep<h_local_pseudo_function<T> >::evaluate_on_side (
    const geo_element&           L,
    const side_information_type& sid,
    std::vector<result_type>&    value) const
{
  evaluate (base::get_side(L, sid), value);
}
// ----------------------------------------------------------------------------
// 1.5) penalty() class-function
// ----------------------------------------------------------------------------
template<class T>
field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >::field_expr_v2_nonlinear_terminal_function_rep(
    const function_type&)
 : base()
{
}
template<class T>
field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >::field_expr_v2_nonlinear_terminal_function_rep(
    const field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >& x)
 : base(x)
{
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >::initialize (
    const geo_basic<float_type,memory_type>& omega, 
    const quadrature<float_type>&            quad) const
{
  return base::initialize (omega, quad);
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >::initialize (
    const space_basic<float_type,memory_type>& Xh) const
{
  return base::initialize (Xh);
}
template<class T>
T
field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >::evaluate_measure (
    const geo_element&        K) const
{
  base::evaluate_initialize (K);
  size_type     d = base::_omega.dimension();
  size_type map_d = K.dimension();
  reference_element hat_K = K.variant();
  tensor_basic<T> DF;
  size_type nq = base::_piola_ops.pointset_size(hat_K);
  T meas_max_K = 0;
  for (size_type q = 0; q < nq; ++q) {
    jacobian_piola_transformation (base::_omega, base::_piola_ops, K, base::_dis_inod, q, DF);
    T meas_K = det_jacobian_piola_transformation (DF, d, map_d);
    meas_max_K = std::max (meas_max_K, abs(meas_K));
  }
  return meas_max_K;
}
template<class T>
void
field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >::evaluate_internal (
    const geo_element&        K,
    const geo_element&        L,
    std::vector<result_type>& value) const
{
  // meas(partial L)/meas(L)  ~= nside*meas(K)/meas(L) : TODO improve it
  std::vector<result_type> value_side;
  T meas_side_K = evaluate_measure (K);
  T meas_K      = evaluate_measure (L);
  reference_element hat_K = K.variant();
  size_type n_sides = L.n_subgeo (K.dimension());
  base::evaluate_initialize (K);
  value.resize (base::_piola_ops.pointset_size(hat_K));
  for (size_type loc_idof = 0, loc_ndof = value.size(); loc_idof < loc_ndof; ++loc_idof) {
    value[loc_idof] = n_sides*meas_side_K/meas_K;
  }
}
template<class T>
bool
field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >::evaluate (
    const geo_element&           K,
    std::vector<result_type>&    value) const
{
  size_type L_dis_ie0 = K.master(0);
  size_type L_dis_ie1 = K.master(1);
  size_type L_map_d = K.dimension() + 1;
  check_macro (L_dis_ie0 != std::numeric_limits<size_type>::max(),
      "unexpected isolated mesh side K.dis_ie="<<K.dis_ie());
  if (L_dis_ie1 == std::numeric_limits<size_type>::max()) {
    // K is a boundary side
    const geo_element& L0 = base::_omega.dis_get_geo_element (L_map_d, L_dis_ie0);
    evaluate_internal (K, L0, value);
    return true;
  }
  // K is an internal side
  const geo_element& L0 = base::_omega.dis_get_geo_element (L_map_d, L_dis_ie0);
  const geo_element& L1 = base::_omega.dis_get_geo_element (L_map_d, L_dis_ie1);
  std::vector<result_type> value1;
  evaluate_internal (K, L0, value);
  evaluate_internal (K, L1, value1);
  for (size_type loc_idof = 0, loc_ndof = value.size(); loc_idof < loc_ndof; ++loc_idof) {
    value[loc_idof] = std::max (value [loc_idof], value1 [loc_idof]);
  }
  return true;
}
template<class T>
void
field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >::evaluate_on_side (
    const geo_element&           L,
    const side_information_type& sid,
    std::vector<result_type>&    value) const
{
  evaluate (base::get_side(L, sid), value);
}
// ----------------------------------------------------------------------------
// 2) field and such
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// 2.1) base class for the field family
// ----------------------------------------------------------------------------
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_base_rep<T,M>::initialize (
    const geo_basic<T,M>& omega,
    const quadrature<T>&  quad) const
{
    check_macro (_uh.get_geo().get_background_geo().name() == omega.get_background_geo().name(),
	"incompatible field on " << _uh.get_geo().get_background_geo().name() << " for evaluation on " << omega.get_background_geo().name());
    _omega = omega;
    _bops.set (quad, _uh.get_space().get_numbering().get_basis());
    _uh.dis_dof_update();
    _use_dom2bgd = (_omega.variant()        == geo_abstract_base_rep<T>::geo_domain &&
                    _uh.get_geo().variant() != geo_abstract_base_rep<T>::geo_domain);
    _use_bgd2dom = (_omega.variant()        != geo_abstract_base_rep<T>::geo_domain &&
                    _uh.get_geo().variant() == geo_abstract_base_rep<T>::geo_domain);
    _have_dg_on_sides = _uh.get_space().get_numbering().is_discontinuous()
                      && _omega.variant() == geo_abstract_base_rep<T>::geo_domain_indirect
                      && 1 + omega.map_dimension() == omega.get_background_geo().map_dimension();
    return true;
}
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_base_rep<T,M>::initialize (
    const space_basic<T,M>& Xh) const
{
    // Xh: only Lagrange nodal basis are supported here
    check_macro (_uh.get_geo().get_background_geo().name() == Xh.get_geo().get_background_geo().name(),
	"incompatible field on " << _uh.get_geo().get_background_geo().name() << " for evaluation on " << Xh.get_geo().get_background_geo().name());
    _omega = Xh.get_geo();
    _bops.set (Xh.get_numbering().get_basis(), _uh.get_space().get_numbering().get_basis());
    _uh.dis_dof_update();
    _use_dom2bgd = (_omega.variant()        == geo_abstract_base_rep<T>::geo_domain &&
                    _uh.get_geo().variant() != geo_abstract_base_rep<T>::geo_domain);
    _use_bgd2dom = (_omega.variant()        != geo_abstract_base_rep<T>::geo_domain &&
                    _uh.get_geo().variant() == geo_abstract_base_rep<T>::geo_domain);
    _have_dg_on_sides = _uh.get_space().get_numbering().is_discontinuous()
                      && (_use_dom2bgd || _omega.variant() == geo_abstract_base_rep<T>::geo_domain_indirect)
                      && 1 + _omega.map_dimension() == _omega.get_background_geo().map_dimension();
    return true;
}
template<class T, class M>
void
field_expr_v2_nonlinear_terminal_field_base_rep<T,M>::evaluate_initialize (
  const geo_element& K) const
{
    if (_have_dg_on_sides && K.dimension() == _omega.map_dimension()) {
      // omits "inner(u)" on a sides-domain => assume a boundary side
      const geo_element* bgd_K_ptr = 0;
      if (_use_dom2bgd) {
        // get bgd_K on background mesh: for DG-interpolate on a boundary domain
        bgd_K_ptr = &(_omega.dom2bgd_geo_element (K));
      } else {
        bgd_K_ptr = &K;
      }
      const geo_element& bgd_K = *bgd_K_ptr;
      size_type L_map_d = bgd_K.dimension() + 1;
      check_macro (bgd_K.master(1) == std::numeric_limits<size_type>::max(),
          "unexpected non-boundary side K.dis_ie="<<bgd_K.dis_ie());
      size_type L_dis_ie = bgd_K.master(0);
      check_macro (L_dis_ie != std::numeric_limits<size_type>::max(),
          "unexpected isolated side K.dis_ie="<<bgd_K.dis_ie());
      const geo_element* L_ptr = 0;
      if (_use_dom2bgd) {
        L_ptr = &(_omega.get_background_geo().dis_get_geo_element (L_map_d, L_dis_ie));
      } else {
        L_ptr = &(_omega.dis_get_geo_element (L_map_d, L_dis_ie));
      }
      const geo_element& L = *L_ptr;
      _tilde_L = L.variant();
      side_information_type sid;
      L.get_side_informations (bgd_K, sid);
      _bops.restrict_on_side (_tilde_L, sid);
      _uh.get_space().dis_idof (L, _dis_idof);
    } else if (_use_bgd2dom) {        // get dom_K from background mesh:
      const geo_element& dom_K = _uh.get_geo().bgd2dom_geo_element (K);
      _uh.get_space().dis_idof (dom_K, _dis_idof);
    } else if (_use_dom2bgd) { // get bgd_K on background mesh:
      const geo_element& bgd_K = _omega.dom2bgd_geo_element (K);
      _uh.get_space().dis_idof (bgd_K, _dis_idof);
    } else {
      // usual case: use K directly
        _uh.get_space().dis_idof (K, _dis_idof);
    }
}
template<class T, class M>
void
field_expr_v2_nonlinear_terminal_field_base_rep<T,M>::evaluate_on_side_initialize (
  const geo_element&           L,
  const side_information_type& sid) const
{
    reference_element tilde_L = L.variant();
    _bops.restrict_on_side (tilde_L, sid);
    _uh.get_space().dis_idof (L, _dis_idof);
}
template<class T, class M>
const geo_element& 
field_expr_v2_nonlinear_terminal_field_base_rep<T,M>::get_side (
  const geo_element&           L,
  const side_information_type& sid) const
{
  return global_get_side (_omega, L, sid);
}
// ----------------------------------------------------------------------------
// 2.2) field
// ----------------------------------------------------------------------------
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_rep<T,M>::initialize (
    const geo_basic<float_type,memory_type>& omega,
    const quadrature<float_type>&            quad) const
{
  return base::initialize (omega, quad);
}
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_rep<T,M>::initialize (
    const space_basic<float_type,memory_type>& Xh) const 
{
  return base::initialize (Xh);
}
template<class T, class M>
template<class Result>
bool
field_expr_v2_nonlinear_terminal_field_rep<T,M>::evaluate ( 
    const geo_element&   K,
    std::vector<Result>& value) const
{
    base::evaluate_initialize (K);
    reference_element hat_K = K.variant();
    if (base::_have_dg_on_sides && K.dimension() == base::_omega.map_dimension()) {
      hat_K = base::_tilde_L;
    }
    value.resize (base::_bops.pointset_size(hat_K));
    size_type q = 0;
    for (typename std::vector<Result>::iterator
	iter = value.begin(),
	last = value.end(); iter != last; ++iter, ++q) {
      general_field_evaluate (base::_uh, base::_bops, hat_K, base::_dis_idof, q, *iter);
    }
    return true;
}
template<class T, class M>
template<class Result>
void 
field_expr_v2_nonlinear_terminal_field_rep<T,M>::evaluate_on_side (
    const geo_element&           L,
    const side_information_type& sid,
    std::vector<Result>&         value) const
{
  base::evaluate_on_side_initialize (L, sid);
  evaluate (L, value);
}
// ---------------------------------------------------------------------------
// 2.3) grad of a field
// ---------------------------------------------------------------------------
template<class T, class M>
space_constant::valued_type 
field_expr_v2_nonlinear_terminal_field_grad_rep<T,M>::valued_tag() const
{
  space_constant::valued_type uh_valued_tag = base::_uh.valued_tag();
  switch (uh_valued_tag) {
    case space_constant::scalar: return space_constant::vector;
    case space_constant::vector: return space_constant::unsymmetric_tensor;
    case space_constant::tensor: 
    case space_constant::unsymmetric_tensor:
                                 return space_constant::tensor3;
    default:
      error_macro ("unsupported "<<uh_valued_tag<< "-valued field for the grad() operator");
      return space_constant::last_valued;
  } 
}
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_grad_rep<T,M>::initialize (
    const geo_basic<float_type,memory_type>& omega,
    const quadrature<float_type>&            quad) const
{
    base::initialize (omega, quad);
    bool ignore_sys_coord = false;
    _u_test.initialize (omega, quad, ignore_sys_coord);
    return true;
}
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_grad_rep<T,M>::initialize (
    const space_basic<float_type,memory_type>& Xh) const 
{
    base::initialize (Xh);
    bool ignore_sys_coord = false;
    _u_test.initialize (Xh, ignore_sys_coord);
    return true;
}
template<class T, class M>
template<class ValueType>
bool
field_expr_v2_nonlinear_terminal_field_grad_rep<T,M>::evaluate ( 
    const geo_element&      K,
    std::vector<ValueType>& value) const
{
    base::evaluate_initialize (K);
    reference_element hat_K = K.variant();
    size_type nq = _u_test.element_initialize (K);
    value.resize (nq);
    size_type loc_ndof = base::_dis_idof.size();
    std::vector<T> udof (loc_ndof);
    for (size_type loc_idof = 0; loc_idof < loc_ndof; ++loc_idof) {
      udof[loc_idof] = base::_uh.dis_dof (base::_dis_idof[loc_idof]);
    }
    std::vector<ValueType> grad_basis (loc_ndof);
    for (size_type q = 0; q != nq; ++q) {
      _u_test.grad_basis_evaluate (hat_K, q, _opt, grad_basis);
      ValueType sum (T(0.));
      for (size_type loc_idof = 0; loc_idof < loc_ndof; ++loc_idof) {
        sum = sum + udof[loc_idof]*grad_basis[loc_idof];
      }
      value[q] = sum;
    }
    return true;
}
// ---------------------------------------------------------------------------
// 2.4) div of a field
// ---------------------------------------------------------------------------
template<class T, class M>
space_constant::valued_type 
field_expr_v2_nonlinear_terminal_field_div_rep<T,M>::valued_tag() const
{
  space_constant::valued_type uh_valued_tag = base::_uh.valued_tag();
  switch (uh_valued_tag) {
    case space_constant::vector: return space_constant::scalar;
    case space_constant::tensor:
    case space_constant::unsymmetric_tensor:
				 return space_constant::vector;
    default:
      error_macro ("unsupported "<<uh_valued_tag<< "-valued field for the div() operator");
      return space_constant::last_valued;
  } 
}
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_div_rep<T,M>::initialize (
    const geo_basic<float_type,memory_type>& omega,
    const quadrature<float_type>&            quad) const
{
    base::initialize (omega, quad);
    bool ignore_sys_coord = false;
    _u_test.initialize (omega, quad, ignore_sys_coord);
    return true;
}
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_div_rep<T,M>::initialize (
    const space_basic<float_type,memory_type>& Xh) const 
{
    base::initialize (Xh);
    bool ignore_sys_coord = false;
    _u_test.initialize (Xh, ignore_sys_coord);
    return true;
}
template<class T, class M, class ValueType>
static
void
div_evaluate_aux (
    const field_expr_v2_nonlinear_terminal_field_div_rep<T,M>& obj,
    const geo_element&                            K,
    std::vector<ValueType>&                       value)
{
    typedef geo_element::size_type size_type;
    obj.evaluate_initialize (K);
    reference_element hat_K = K.variant();
    size_type nq = obj._u_test.element_initialize (K);
    value.resize (nq);
    std::vector<ValueType> div_basis (obj._dis_idof.size());
    for (size_type q = 0; q != nq; ++q) {
      obj._u_test.div_basis_evaluate (hat_K, q, obj._opt, div_basis);
      ValueType sum (0.);
      for (size_type loc_idof = 0, loc_ndof = div_basis.size(); loc_idof < loc_ndof; ++loc_idof) {
        const T& udof = obj._uh.dis_dof (obj._dis_idof[loc_idof]);
        sum = sum + udof*div_basis[loc_idof];
      }
      value[q] = sum;
    }
}
// div(vector) -> scalar
template<class T, class M>
static
void
div_evaluate (
    const field_expr_v2_nonlinear_terminal_field_div_rep<T,M>& obj,
    const geo_element&                            K,
    std::vector<T>&                               value)
{
  div_evaluate_aux (obj, K, value);
}
// div(tensor) -> vector
template<class T, class M>
static
void
div_evaluate (
    const field_expr_v2_nonlinear_terminal_field_div_rep<T,M>& obj,
    const geo_element&                            K,
    std::vector<point_basic<T> >&                 value)
{
  div_evaluate_aux (obj, K, value);
}
template<class T, class M>
static
void
div_evaluate (
    const field_expr_v2_nonlinear_terminal_field_div_rep<T,M>& obj,
    const geo_element&                            K,
    std::vector<tensor_basic<T> >&                value)
{
  fatal_macro ("div(tensor3): not yet supported");
}
template<class T, class M>
template<class ValueType>
bool
field_expr_v2_nonlinear_terminal_field_div_rep<T,M>::evaluate ( 
    const geo_element&      K,
    std::vector<ValueType>& value) const
{
  div_evaluate (*this, K, value);
  return true;
}
// ---------------------------------------------------------------------------
// 2.5) curl of a field
// ---------------------------------------------------------------------------
template<class T, class M>
space_constant::valued_type 
field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>::valued_tag() const
{
  space_constant::valued_type uh_valued_tag = base::_uh.valued_tag();
  switch (uh_valued_tag) {
    case space_constant::scalar:    return space_constant::vector;
    case space_constant::vector: {
        size_type d = base::_uh.get_geo().dimension();
        return (d==2) ? space_constant::scalar : space_constant::vector;
    }
    default:
      error_macro ("unsupported "<<uh_valued_tag<< "-valued field for the curl() operator");
      return space_constant::last_valued;
  } 
}
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>::initialize (
    const geo_basic<float_type,memory_type>& omega,
    const quadrature<float_type>&            quad) const
{
    base::initialize (omega, quad);
    bool ignore_sys_coord = false;
    _u_test.initialize (omega, quad, ignore_sys_coord);
    return true;
}
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>::initialize (
    const space_basic<float_type,memory_type>& Xh) const 
{
    base::initialize (Xh);
    bool ignore_sys_coord = false;
    _u_test.initialize (Xh, ignore_sys_coord);
    return true;
}
// curl(vector) -> scalar, 2D
// curl(scalar) -> vector, 2D
// curl(vector) -> vector, 3D
template<class T, class M, class ValueType>
static
bool
curl_evaluate_aux (
    const field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>& obj,
    const geo_element&                             K,
    std::vector<ValueType>&                        value)
{
    typedef geo_element::size_type size_type;
    obj.evaluate_initialize (K);
    reference_element hat_K = K.variant();
    size_type nq = obj._u_test.element_initialize (K);
    value.resize (nq);
    std::vector<ValueType> curl_basis (obj._dis_idof.size());
    for (size_type q = 0; q != nq; ++q) {
      obj._u_test.curl_basis_evaluate (hat_K, q, obj._opt, curl_basis);
      ValueType sum (0.);
      for (size_type loc_idof = 0, loc_ndof = curl_basis.size(); loc_idof < loc_ndof; ++loc_idof) {
        const T& udof = obj._uh.dis_dof (obj._dis_idof[loc_idof]);
        sum = sum + udof*curl_basis[loc_idof];
      }
      value[q] = sum;
    }
    return true;
}
template<class T, class M>
static
void
curl_evaluate (
    const field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>& obj,
    const geo_element&                             K,
    std::vector<T>&                                value)
{
  curl_evaluate_aux (obj, K, value);
}
template<class T, class M>
static
void
curl_evaluate (
    const field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>& obj,
    const geo_element&                             K,
    std::vector<point_basic<T> >&                  value)
{
  curl_evaluate_aux (obj, K, value);
}
// curl(tensor3) -> tensor
template<class T, class M>
static
void
curl_evaluate (
    const field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>& obj,
    const geo_element&                            K,
    std::vector<tensor_basic<T> >&                value)
{
  fatal_macro ("curl() -> tensor: unsupported");
}
// curl(tensor4) -> tensor3
template<class T, class M>
static
void
curl_evaluate (
    const field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>& obj,
    const geo_element&                            K,
    std::vector<tensor3_basic<T> >&               value)
{
  fatal_macro ("curl() -> tensor3: unsupported");
}
// curl(tensor5) -> tensor4
template<class T, class M>
static
void
curl_evaluate (
    const field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>& obj,
    const geo_element&                            K,
    std::vector<tensor4_basic<T> >&               value)
{
  fatal_macro ("curl() -> tensor3: unsupported");
}
template<class T, class M>
template<class ValueType>
bool
field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>::evaluate ( 
    const geo_element&      K,
    std::vector<ValueType>& value) const
{
  curl_evaluate (*this, K, value);
  return true;
}
// ----------------------------------------------------------------------------
// 2.6) field jump
// ----------------------------------------------------------------------------
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_dg_rep<T,M>::initialize (
    const geo_basic<float_type,memory_type>& omega,
    const quadrature<float_type>&            quad) const
{
  return _expr0.initialize (omega, quad) &&
         _expr1.initialize (omega, quad);
}
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_dg_rep<T,M>::initialize (
    const space_basic<float_type,memory_type>& Xh) const 
{
  return _expr0.initialize (Xh) &&
         _expr1.initialize (Xh);
}
template<class T, class M>
template<class Result>
bool
field_expr_v2_nonlinear_terminal_field_dg_rep<T,M>::evaluate ( 
    const geo_element&   K,
    std::vector<Result>& value) const
{
  size_type L_map_d = K.dimension() + 1;
  size_type L_dis_ie0, L_dis_ie1;
  side_information_type sid0, sid1;

  L_dis_ie0 = K.master(0);
  L_dis_ie1 = K.master(1);
  check_macro (L_dis_ie0 != std::numeric_limits<size_type>::max(),
      "unexpected isolated mesh side K.dis_ie="<<K.dis_ie());
  if (L_dis_ie1 == std::numeric_limits<size_type>::max()) {
    // K is a boundary side
    const geo_element& L0 = _expr0.get_geo().dis_get_geo_element (L_map_d, L_dis_ie0);
    L0.get_side_informations (K, sid0);
    _expr0.evaluate_on_side (L0, sid0, value);
    // average (i.e. _c0==0.5): fix it on the boundary where c0=1 : average(v)=v on the boundary
    Float c0 = (_c0 != 0.5) ? _c0 : 1;
    for (size_type loc_idof = 0, loc_ndof = value.size(); loc_idof < loc_ndof; ++loc_idof) {
      value[loc_idof] = c0*value[loc_idof];
    }
    return true;
  }
  // K is an internal side
  std::vector<Result> value1;
  const geo_element& L0 = _expr0.get_geo().dis_get_geo_element (L_map_d, L_dis_ie0);
  const geo_element& L1 = _expr1.get_geo().dis_get_geo_element (L_map_d, L_dis_ie1);
  L0.get_side_informations (K, sid0);
  L1.get_side_informations (K, sid1);
  _expr0.evaluate_on_side (L0, sid0, value);
  _expr1.evaluate_on_side (L1, sid1, value1);
  for (size_type loc_idof = 0, loc_ndof = value.size(); loc_idof < loc_ndof; ++loc_idof) {
    value[loc_idof] = _c0*value[loc_idof] + _c1*value1[loc_idof];
  }
  return true;
}
// ---------------------------------------------------------------------------
// 3) contains a convected field, as compose(uh,X) where X is a characteristic
// ---------------------------------------------------------------------------
template<class T, class M>
field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>::field_expr_v2_nonlinear_terminal_field_o_characteristic_rep (
    const field_basic<T,M>&          uh,
    const characteristic_basic<T,M>& X)
  : _uh (uh), 
    _X (X),
    _is_initialized (false),
    _quad(),
    _bops(),
    _uq(),
    _uq_K(),
    _start_q(0)
{
}
template<class T, class M>
field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>::field_expr_v2_nonlinear_terminal_field_o_characteristic_rep (
    const field_o_characteristic<T,M>& uoX)
  : _uh (uoX.get_field()), 
    _X (uoX.get_characteristic()),
    _is_initialized (false),
    _quad(),
    _bops(),
    _uq(),
    _start_q(0)
{
}
template<class T, class M>
void
interpolate_pass2_valued (
    const field_basic<T,M>&                uh,
    const disarray<point_basic<T>,M>&      x,
    const disarray<index_set,M>&           ie2dis_ix, // K -> list of ix
    const disarray<point_basic<T>,M>&      hat_y,     // ix -> hat_y
          disarray<T,M>&                   ux);

template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>::initialize (
    const geo_basic<float_type,memory_type>& omega,
    const quadrature<float_type>&            quad) const
{
    // TODO: omega not used here...
    // handle the case when omega (destination) is different from _X.get_displacement.get_geo (origin)
    // as in mesh adaptation loops
    _uh.dis_dof_update();

    quadrature_option qopt = quad.get_options();
 
    // coq is stored in _X : it is independent of _uh ; depend only of _uh.space and qopt
    const characteristic_on_quadrature<T,M> coq = _X.get_pre_computed (_uh.get_space(), _X.get_displacement(), qopt);
    const characteristic_on_quadrature_rep<T,M>& coq_r = coq.data();
    _uq.resize (coq_r._yq.ownership());
    interpolate_pass2_valued (
      _uh, 
      coq_r._yq,
      coq_r._ie2dis_ix,
      coq_r._hat_y,
      _uq);
    _uq_K = _uq.begin();
    _start_q = 0;

    _quad = coq_r._quad;
    _bops.set (coq_r._quad, _uh.get_space().get_numbering().get_basis());
    _is_initialized = true;
    return true;
}
template<class T, class M>
bool
field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>::initialize ( 
    const space_basic<float_type,memory_type>& Xh) const
{
    // Xh: only Lagrange nodal basis are supported here
    _bops.set (Xh.get_numbering().get_basis(), _uh.get_space().get_numbering().get_basis());
    _uh.dis_dof_update();
    _start_q = 0;
    _is_initialized = true;
    //TODO: require to evaluate uoX at all Lagrange nodes of Xh
    // instead of all quadrature nodes for Gauus-Lobatto
    fatal_macro ("characteristic: not yet for interpolate");
    return true;
}
// general case:
template<class T, class M, class Result>
void
set_value (
// input:
  const field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>& obj,
  const geo_element&                  K, 
  typename disarray<T,M>::const_iterator uq_K,
// modified:
  geo_element::size_type&             start_q,
  std::vector<Result>&                value)
{
    fatal_macro ("characteristic: unexpected valued field with value_type="<<typename_macro(Result));
}
// scalar-valued case:
template<class T, class M>
static
void
set_value (
  const field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>& obj,
  const geo_element&                  K, 
  std::vector<T>&                     value)
{
  for (geo_element::size_type loc_idof = 0, loc_ndof = value.size(); loc_idof < loc_ndof; loc_idof++, obj._start_q++) {
    value [loc_idof] = obj._uq_K [obj._start_q];
  }
}
// vector-valued case:
template<class T, class M>
static
void
set_value (
  const field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>& obj,
  const geo_element&                  K, 
  std::vector<point_basic<T> >&       value)
{
  typedef geo_element::size_type size_type;
  const geo_basic<T,M>& omega = obj._uh.get_geo(); // TODO: when another destination mesh omega, different from _uh.geo
  size_type nloc_comp_q    = obj._quad.size(K);
  size_type comp_nq        = omega.size()*nloc_comp_q;
  size_type n_comp         = obj._uh.size();
  for (geo_element::size_type loc_comp_idof = 0, loc_comp_ndof = value.size(); loc_comp_idof < loc_comp_ndof; loc_comp_idof++) {
    for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
      size_type q = obj._start_q + loc_comp_idof + i_comp*comp_nq;
      value [loc_comp_idof][i_comp] = obj._uq_K [q];
    }
  }
  obj._start_q += nloc_comp_q;
}
// tensor-valued case:
template<class T, class M>
static
void
set_value (
  const field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>& obj,
  const geo_element&                  K, 
  std::vector<tensor_basic<T> >&      value)
{
  typedef geo_element::size_type size_type;
  const geo_basic<T,M>& omega = obj._uh.get_geo(); // TODO: when another destination mesh omega, different from _uh.geo
  size_type nloc_comp_q    = obj._quad.size(K);
  size_type comp_nq        = omega.size()*nloc_comp_q;
  size_type n_comp         = obj._uh.size();
  space_constant::coordinate_type sys_coord = omega.coordinate_system();
  for (geo_element::size_type loc_comp_idof = 0, loc_comp_ndof = value.size(); loc_comp_idof < loc_comp_ndof; loc_comp_idof++) {
    tensor_basic<T>& vi = value [loc_comp_idof];
    for (size_type ij_comp = 0; ij_comp < n_comp; ij_comp++) {
      size_type q = obj._start_q + loc_comp_idof + ij_comp*comp_nq;
      std::pair<size_type,size_type> ij
          = space_constant::tensor_subscript (space_constant::tensor, sys_coord, ij_comp);
      vi(ij.first,ij.second) = obj._uq_K [q];
      if (ij.first != ij.second) { vi(ij.second,ij.first) = vi(ij.first,ij.second); }
    }
  }
  obj._start_q += nloc_comp_q;
}
// tensor3-valued case:
template<class T, class M>
static
void
set_value (
  const field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>& obj,
  const geo_element&                  K, 
  std::vector<tensor3_basic<T> >&     value)
{
    fatal_macro ("characteristic: not yet for tensor3-valued fields");
}
// tensor4-valued case:
template<class T, class M>
static
void
set_value (
  const field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>& obj,
  const geo_element&                  K, 
  std::vector<tensor4_basic<T> >&     value)
{
    fatal_macro ("characteristic: not yet for tensor4-valued fields");
}
template<class T, class M>
template<class Result>
bool
field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>::evaluate (
    const geo_element&   K, 
    std::vector<Result>& value) const
{
    reference_element hat_K = K.variant();
    value.resize (_bops.pointset_size(hat_K));
    set_value (*this, K, value);
    return true;
}
// ----------------------------------------------------------------------------
// instanciation in library (v2)
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation_base_seq_only(T) 			      		   \
template class field_expr_v2_nonlinear_terminal_function_base_rep<T>; 	      		   \
template class field_expr_v2_nonlinear_terminal_function_rep< normal_pseudo_function<T> >; \
template class field_expr_v2_nonlinear_terminal_function_rep<h_local_pseudo_function<T> >; \
template class field_expr_v2_nonlinear_terminal_function_rep<penalty_pseudo_function<T> >; \

// extra when Result=tensor3, tensor4
#define _RHEOLEF_instanciation_base_both_members_full(T,M,Result) 				\
template bool field_expr_v2_nonlinear_terminal_field_rep<T,M>::evaluate<Result > ( 		\
    		const geo_element&, std::vector<Result >&) const;				\
template void field_expr_v2_nonlinear_terminal_field_rep<T,M>::evaluate_on_side<Result > ( 	\
    		const geo_element& K, const side_information_type& sid, 			\
		std::vector<Result>& value) const;						\
template bool field_expr_v2_nonlinear_terminal_field_grad_rep<T,M>::evaluate<Result > ( 	\
    		const geo_element&, std::vector<Result >&) const;				\
template bool field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>::evaluate<Result > ( 	\
    		const geo_element&, std::vector<Result >&) const;				\
template bool field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>::evaluate<Result > ( \
    		const geo_element&, std::vector<Result >&) const;				\
template bool field_expr_v2_nonlinear_terminal_field_dg_rep<T,M>::evaluate<Result > ( 		\
    		const geo_element&, std::vector<Result >&) const;				\

#define _RHEOLEF_instanciation_base_both_members_mini(T,M,Result) 				\
	_RHEOLEF_instanciation_base_both_members_full(T,M,Result)				\
template bool field_expr_v2_nonlinear_terminal_field_div_rep<T,M>::evaluate<Result > ( 		\
    		const geo_element&, std::vector<Result >&) const;				\

#define _RHEOLEF_instanciation_base_both(T,M) 					\
template class field_expr_v2_nonlinear_terminal_field_rep<T,M>;			\
template class field_expr_v2_nonlinear_terminal_field_grad_rep<T,M>;		\
template class field_expr_v2_nonlinear_terminal_field_div_rep<T,M>;		\
template class field_expr_v2_nonlinear_terminal_field_curl_rep<T,M>;		\
template class field_expr_v2_nonlinear_terminal_field_dg_rep<T,M>;		\
template class field_expr_v2_nonlinear_terminal_field_o_characteristic_rep<T,M>; \
        _RHEOLEF_instanciation_base_both_members_mini(T,M,T) 			\
        _RHEOLEF_instanciation_base_both_members_mini(T,M,point_basic<T>) 	\
        _RHEOLEF_instanciation_base_both_members_mini(T,M,tensor_basic<T>) 	\
        _RHEOLEF_instanciation_base_both_members_full(T,M,tensor3_basic<T>) 	\
        _RHEOLEF_instanciation_base_both_members_full(T,M,tensor4_basic<T>)	\

#define _RHEOLEF_instanciation_seq(T)	 					\
        _RHEOLEF_instanciation_base_seq_only(T) 				\
        _RHEOLEF_instanciation_base_both(T,sequential)				\

#define _RHEOLEF_instanciation_dis(T) 						\
        _RHEOLEF_instanciation_base_both(T,distributed)				\

_RHEOLEF_instanciation_seq(Float)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation_dis(Float)
#endif // _RHEOLEF_HAVE_MPI

}} // namespace rheolef::details
