#!/bin/bash

set -e

# This script is used by nomeata to 
# * Create a tarball from the upstream git branch 
# * Insert it into pristine tar
# * Merge it into Debian
# * Create a new debian/changelog entry 

# It requires the upstream/$VERSION tag to be set

VERSION=$1

if [ -z "$VERSION" -o "$VERSION" = "--help" ]
then
	echo "Usage:"
	echo "	$0 <version>"
	echo "Whereas a tag upstream/<version> exists"
	exit 0
fi

if [ ! -e .git ]
then
	echo "Please run in the root of the git tree"
	exit 1
fi


TARBALL=../fso-frameworkd_"$VERSION".orig.tar.gz
TAG="upstream/$VERSION"

if !  git tag -l "$TAG" | fgrep -q "$TAG"
then
	echo "Tag $TAG not found"
	exit 1
fi

if ! git status | fgrep -q 'nothing to commit'
then
	echo "Uncommited changes:"
	git status
	exit 1
fi

set -x

git-archive --format=tar --prefix=frameworkd-"$VERSION"/ "$TAG" | \
	gzip -9 > $TARBALL

pristine-tar -m "New upstream tarball

Generated by:
git-archive --format=tar --prefix=frameworkd-$VERSION/ "$TAG" | \
	gzip -9 > ../fso-frameworkd_$VERSION.orig.tar.gz" \
	commit "$TARBALL" "$TAG"

git checkout debian

git merge "$TAG"

# Even if the last entry was UNRELEASED, we want a new one
dch --release-heuristic log --distribution UNRELEASED --newversion "$VERSION-1" \
 -m "New upstream release"

git commit debian/changelog -m "Entry about new upstream version $VERSION"

echo "Done!"
