/*
 * IRiver ifp supporting functions, a basic (primative) API.
 * $Id: ifpline.c,v 1.5 2004/11/18 17:12:37 oakhamg Exp $
 *
 * Copyright (C) Pavel Kriz, 2004; <pavkriz@gybon.cz>
 * Copyright (C) Geoff Oakham, 2004; <oakhamg@users.sourceforge.net>
 * Copyright (C) Joe Roback, 2004; <joe@roback.cc>
 * Copyright (C) Jun Yamashiro, 2004; <yamajun@{ofug.net,cr.ie.u-ryukyu.ac.jp}>
 */

static const char rcsid[] = "$Id: ifpline.c,v 1.5 2004/11/18 17:12:37 oakhamg Exp $";

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <fts.h>
#include <unistd.h>
#include <usb.h>

#include "ifp.h"

#define IS_IFP          0x0
#define IS_MC           0x1
#define MC_INFO         "!!!INFO!!!"

int filestat(const char *name);
char *get_basename(const char *path);
char *ifp_strcasestr(const char *base, const char *target);
void normalize_path(char *path, size_t size);
void remove_double_slash(char *path, size_t size);
void backslash2slash(char *dest, const char *src, size_t count);
void slash2backslash(char *dest, const char *src, size_t count);
void help(char *prog);

static int update_progress(void * context, struct ifp_transfer_status * status) {
	char * verb = context;
	printf("%d %s          \r", (int)(status->batch_bytes), verb);
	fflush(stdout);
	return 0;
}

int upload(struct ifp_device *dev, int argc, char *argv[]) {
    int retval = 0;
    char ifp_path[IFP_MAXPATHLEN];
    char path_buf[PATH_MAX];
    ifp_progress progress = update_progress;
    void * p_context = "sent";

    if (argc < 2) {
	printf("usage: upload localfile ifptarget\n");
	printf("       upload localdir  ifpdir\n");
	return -1;
    }

    switch (filestat(argv[0])) {
    case 0:	// file not exist
	fprintf(stderr, "ifp upload: %s: not found.\n", argv[0]);
	return -1;
	break;

    case 1:	// regular file
	    strncpy(ifp_path, argv[1], IFP_MAXPATHLEN - 1);
	    normalize_path(ifp_path, IFP_MAXPATHLEN);
	    slash2backslash(ifp_path, ifp_path, IFP_MAXPATHLEN);

	    if (ifp_is_dir(dev, ifp_path)) {
		// file to dir
		if (ifp_path[strlen(ifp_path) - 1] != '\\') {
			strncat(ifp_path, "\\", 1);
		}
		strncat(ifp_path, get_basename(argv[0]),
			IFP_MAXPATHLEN - strlen(ifp_path) - 1);

		retval = ifp_upload_file(dev, argv[0], ifp_path, progress, p_context);
	    } else {
		retval = ifp_upload_file(dev, argv[0], ifp_path, progress, p_context);  // file to file
	    }
	break;

    case 2:	// directory
	strncpy(path_buf, argv[0], PATH_MAX - 1);
	normalize_path(path_buf, PATH_MAX);

	strncpy(ifp_path, argv[1], IFP_MAXPATHLEN - 1);
	normalize_path(ifp_path, IFP_MAXPATHLEN);
	slash2backslash(ifp_path, ifp_path, IFP_MAXPATHLEN);

	retval = ifp_upload_dir(dev, path_buf, ifp_path, progress, p_context);
	break;

    default:
	return -1;
	break;
    }

    if (retval <= -2) {
	fprintf(stderr, "ifp upload: Not enough space on device.\n");
	return -1;
    }
	
    return retval;
}

int ifp_put(struct ifp_device *dev, int argc, char *argv[]) {
    struct stat status;
    char *nargv[3];

    if (argc < 2) {
	printf("usage: put localfile\n");
	printf("       put localdir\n");
    }
    nargv[0] = argv[0];
    nargv[1] = argv[1];

    if (stat(argv[1], &status) != -1) {
	if (S_ISDIR(status.st_mode)) {
	    nargv[2] = "/";
	} else {
	    nargv[2] = argv[1];
	}
    } else {
	perror("unable to upload file");
	return -1;
    }

    return upload(dev, 3, nargv);
}


int display_progress(void * context, int bytes) {
	char * verb = context;
	printf("%d %s\r", bytes, verb);
	fflush(stdout);
	return 0;
}

int download(struct ifp_device * dev, int argc, char *argv[]) {
    int i = 0;
    char ifp_path[IFP_MAXPATHLEN];
    char path_buf[PATH_MAX];

    ifp_progress progress = update_progress;
    void * p_context = "received";

    if (argc < 2) {
	printf("usage: download ifpfile localtarget\n");
	printf("       download ifpdir  localdir\n");
	return 1;
    }

    strncpy(ifp_path, argv[0], IFP_MAXPATHLEN - 1);
    normalize_path(ifp_path, IFP_MAXPATHLEN);
    slash2backslash(path_buf, ifp_path, IFP_MAXPATHLEN);
    strncpy(ifp_path, path_buf, IFP_MAXPATHLEN);

    switch (filestat(argv[1])) {
    case 0: // "argv[1]" does not exist. create new file.
	// file to file
	if (ifp_is_dir(dev, ifp_path)) {
	    fprintf(stderr,
		    "ifp download: Cannot copy %s to %s(%s is a directory).\n",
		    ifp_path, argv[1], ifp_path);
	    i = -1;
	    break;
	}

	// This function is never call in mc plugin mode.
	i = ifp_download_file(dev, ifp_path, argv[1], progress, p_context);
	ifp_err_expect(i, i==-ENOENT||i==-EEXIST||i==-ENOSPC||i==IFP_ERR_BAD_FILENAME
	    ||i==IFP_ERR_USER_CANCEL, err, "problem downloading ifp:\\%s to %s",
	    ifp_path, argv[1]);

	break;
    case 1: // regular file
	// file to file(exist)
	if (ifp_is_dir(dev, ifp_path)) {
	    fprintf(stderr,
		    "ifp download: Cannot copy ifp:\\%s to %s (ifp:\\%s is a directory).\n",
		    ifp_path, argv[1], ifp_path);
	    i = -1;
	    break;
	}

	printf("WARNING: \"%s\" exist. Do you continue?: ", argv[1]);
	switch (getc(stdin)) {
	case 'Y':	/* yes */
	case 'y':
	    i = ifp_download_file(dev, ifp_path, argv[1], progress, p_context);
	    ifp_err_expect(i, i==-ENOENT||i==-EEXIST||i==-ENOSPC||i==IFP_ERR_BAD_FILENAME
		||i==IFP_ERR_USER_CANCEL, err, "problem downloading ifp:\\%s overwriting %s",
		ifp_path, argv[1]);

	    break;
	default:	/* no */
	    i = -1;
	    break;
	}
	break;

    case 2: // directory
	if (ifp_is_dir(dev, ifp_path)) {
	    // dir to dir
	    strncpy(path_buf, argv[1], PATH_MAX - 2);
	    strncat(path_buf, "/", 1);
	    strncat(path_buf, get_basename(ifp_path), PATH_MAX - strlen(path_buf) - 1);
	    
	    i = ifp_download_dir(dev, ifp_path, path_buf, progress, p_context);
	    ifp_err_expect(i, i==-ENOENT||i==-EACCES||i==-ENOSPC||i==IFP_ERR_USER_CANCEL,
		err, "problem downloading directory ifp:\\%s to %s", ifp_path, path_buf);
	} else {
	    // file to dir
	    strncpy(path_buf, argv[1], PATH_MAX - 2);
	    strncat(path_buf, "/", 1);
	    strncat(path_buf, get_basename(ifp_path), PATH_MAX - strlen(path_buf) - 1);

	    i = ifp_download_file(dev, ifp_path, path_buf, progress, p_context);
	    ifp_err_expect(i, i==-ENOENT||i==-EEXIST||i==-ENOSPC||i==IFP_ERR_BAD_FILENAME
	        ||i==IFP_ERR_USER_CANCEL, err, "problem downloading file ifp:\\%s as %s",
		ifp_path, path_buf);
	}
	break;

    default:
	return -1;
	break;
    }

err:
    return i;
}

int ifp_get(struct ifp_device *dh, int argc, char *argv[]) {
    char *nargv[2];

    if (argc < 2) {
	printf("usage: get ifpfile\n");
	printf("       get ifpdir\n");
    }

    nargv[0] = argv[0];

    if (ifp_is_dir(dh, argv[0])) {
	nargv[1] = ".";
    } else {
	nargv[1] = argv[0];
    }

    return download(dh, 2, nargv);
}

static int print_basic(void * context, int type, const char * f, int size) {
	FILE * o = context;
	if (type == IFP_DIR) {
		fprintf(o, "d %s\n", f);
	} else {
		fprintf(o, "f %s\t(size %d)\n", f, size);
	}
	return 0;
}

int print_directory(struct ifp_device *dev, const char *f) {
	int i = 0;
	unsigned char buf[IFP_MAXPATHLEN];

	strncpy(buf, f, IFP_MAXPATHLEN);
	normalize_path(buf, IFP_MAXPATHLEN);
	slash2backslash(buf, buf, IFP_MAXPATHLEN);

	i = ifp_list_dirs(dev, buf, print_basic, stdout);
	ifp_err_expect(i, i==-ENOENT, err, "problem getting directory listing");

err:
	return i;
}

//directory listing
int fancy_print(char special, char read, char write, char execute,
	int filesize, const char * filename)
{
	int i = 0;
    	char b[IFP_MAXPATHLEN];
	backslash2slash(b, filename, IFP_MAXPATHLEN);

	printf("%c%c%c%c%c%c%c%c%c%c  1 root  root %8d Jan  1 1981 %s\n",
		special,
		read, write, execute,
		read, write, execute,
		read, write, execute,
		filesize, b);

	return i;
}

int list_all_files(struct ifp_device *dev)
{
	int i=0, e=0;
	struct ifp_treewalk_entry * f = NULL;
	void * d = NULL;

	i = ifp_treewalk_open(dev, "\\", &d);
	ifp_err_jump(i, err0, "couldn't open ifp:\\%s", "\\");
	while ((f = ifp_treewalk_next(d)) != NULL) {
		char t='-', x='-';
		int filesize = -1;
		switch(f->type) {
		case IFP_WALK_FILE:
			t='-';
			x='-';
			filesize = f->filesize;
			break;
		case IFP_WALK_DIR_PRE:
			t='d';
			x='x';
			filesize = 0;
		break;
		}
		switch(f->type) {
		case IFP_WALK_FILE:
		case IFP_WALK_DIR_PRE:
			i = fancy_print(t, 'r', '-', x, filesize, f->path);
			ifp_err_jump(i, err1, "fancy print failed");
			break;
		}
	}

err1:
	e = ifp_treewalk_close(d);
	d = NULL;
	if (e)
		ifp_err_i(e, "problem closing directory list");
err0:
	return i;
}

int ifp_check_connect(struct ifp_device *dh) {
    return (ifp_battery(dh) >= 0);
}

int format_media(struct ifp_device *dev) {
	int i=0;

	printf("WARNING: Do you want to format iFP? [y/N]: ");
	switch (getc(stdin)) {
	case 'Y':	/* yes */
	case 'y':
		break;
	default:	/* no */
		return 1;
		break;
	}

	printf("Formating. Please wait...\n");
	i = ifp_format(dev);
	ifp_err_jump(i, err, "problem formatting");

	printf("Done.\n");

err:
    return i;
}

int firmware_update(struct ifp_device *dev, int argc, char *argv[]) {
	int i = 0;

	if (argc < 1) {
		printf("usage: firmupdate /path/to/FIRMWARE.HEX\n");
		return 1;
	} else {
		printf("WARNING: Do you want to firmware update? [y/N]: ");
		switch (getc(stdin)) {
		case 'Y':   /* yes */
		case 'y':
			break;
		default:    /* no */
			return 1;
			break;
		}
	}

	printf("Upgrading firmware, please wait.\n");
	i = ifp_update_firmware(dev, argv[0], update_progress, "sent");
	if (i == 0) {
		printf("Firmware upload in progress.  Please don't touch the device until after it turns itself off\n");
	} else {
		ifp_err_i(i, "firmware update failed");
	}

	return i;
}

void info(struct ifp_device *dh, FILE *f, int head) {
    int i;
    char buf[255];
    if (head) {
    fprintf(f,"### This file is virtual, it's not saved in your iFP, but generated by driver\n");
    fprintf(f,"### Following information is obtained only once per connection,\n");
    fprintf(f,"###   press enter (run) on this file to get up-to-date information!\n");
    fprintf(f,"\n");
    }
    i = ifp_device_info(dh, buf, sizeof(buf));
    ifp_err_jump(i, err, "can't get info");
    //fprintf(f,"Product:         %10s (firmware %.2f)\n",buf,ifp_firmware_version(dh)/100.0);
    fprintf(f,"%s\n",buf);
    i = ifp_capacity(dh);
    fprintf(f,"Available space: %10d (%.1fMB)\n",i,i/1024.0/1024.0);
    i = ifp_freespace(dh);
    fprintf(f,"Free space:      %10d (%.1fMB)\n",i,i/1024.0/1024.0);
    //fprintf(f,"Battery status:  %10d\n",ifp_battery(dh));
err:
    return;
}

static inline int clean_and_call_path(struct ifp_device * dev, const char * p,
	int (* fn)(struct ifp_device * dev, const char *))
{
	int i = 0;
	unsigned char buf[IFP_MAXPATHLEN];

	strncpy(buf, p, IFP_MAXPATHLEN);
	normalize_path(buf, IFP_MAXPATHLEN);
	slash2backslash(buf, buf, IFP_MAXPATHLEN);

	i = fn(dev, buf);
	if (i) {
		ifp_err("%s, ifp:\\%s", ifp_error_message(i), buf);
	}

	return i;
}

int ifp_rm(struct ifp_device *dev, int argc, char *argv[]) {

    int retval = 0;

    if (argc < 1) {
	printf("usage: rm [-r] file\n");
	return -1;
    } else {
	if (strcmp(*argv, "-r") == 0) {
	    argc--;
	    argv++;
	    if (argc == 0) {
		printf("usage: rm [-r] file\n");
		return -1;
	    } else {
		retval = clean_and_call_path(dev, *argv, ifp_delete_dir_recursive);
	    }

	} else {
	    retval = clean_and_call_path(dev, *argv, ifp_delete);
	}
    }

    return retval;
}

static inline int clean_and_call_arg(struct ifp_device * dev,
	int argc, char *argv[], int (* fn)(struct ifp_device * dev, const char *))
{
	if (argc == 0) {
		ifp_err("no argument");
		return -1;
	}
	return clean_and_call_path(dev, *argv, fn);
}


int ifp_make_dir(struct ifp_device *dev, int argc, char *argv[]) {
	return clean_and_call_arg(dev, argc, argv, ifp_mkdir);
}

int ifp_delete_dir(struct ifp_device *dev, int argc, char *argv[]) {
	return clean_and_call_arg(dev, argc, argv, ifp_rmdir);
}

int tuner_show(struct ifp_device *dev) {
	char buf[IFP_TUNER_PRESET_DATA];
	char label[IFP_TUNER_LABEL + 1];
	int freq;
	int i = 0;
	int j;

	i = ifp_get_tuner_presets(dev, buf, sizeof(buf));
	ifp_err_jump(i, err, "couldn't get presets");

	for (j=0; j!=IFP_PRESET_TOTAL; j++) {
		i = ifp_get_station(j, buf, label, &freq);
		ifp_err_jump(i, err, "couldn't get station %d", j);

		printf("%2d: %-6.6s %3d.%02dMHz\n", j+1, label, freq/100, freq%100);
	}


err:
	return i;
}

int tuner_set(struct ifp_device *dev, int argc, char *argv[]) {
	char buf[IFP_TUNER_PRESET_DATA];
	const char * label = NULL;
	double d_freq;
	int freq;
	int i = 0;
	int j;

	if (argc < 3) {
		printf("usage: setpreset <slot> <label> <frequency>\n");
		printf("  (eg: setpreset 14 wkrp 101.1)\n");
		return -1;
	}

	label = argv[1];
	j = atoi(argv[0]);
	if (j <= 0 || j > 20) {
		printf("usage: setpreset <slot> <label> <frequency>\n");
		printf("  (eg: setpreset 14 wkrp 101.1)\n");
		return -1;
	} else {
		j--;
	}

	d_freq = strtod(argv[2], NULL);
	freq = (int)(d_freq*100.0 + 0.5);
	if (d_freq == 0.0 || freq < IFP_FREQ_MIN || freq > IFP_FREQ_MAX) {
		printf("usage: setpreset <slot> <label> <frequency>\n");
		printf("  (eg: setpreset 14 wkrp 101.1)\n");
		return -1;
	}
	//printf("setting station %d to %d 10xkHz (%s)\n", j, freq, label);


	i = ifp_get_tuner_presets(dev, buf, sizeof(buf));
	ifp_err_jump(i, err, "couldn't get presets");

	i = ifp_set_station(j, buf, label, freq);
	ifp_err_jump(i, err, "couldn't set station %d", j);

	i = ifp_set_tuner_presets(dev, buf, sizeof(buf));
	ifp_err_jump(i, err, "couldn't save presets");
err:
	return i;
}

/****** support functions ************/
int filestat(const char *name) {
    struct stat status;

    if (stat(name, &status) < 0) {
	if (errno == ENOENT) {
	    // file not exist.
	    return 0;
	}
	perror(name);
	return -1;	// error on stat()
    } else {
	if (S_ISREG(status.st_mode)) {
	    return 1;	// Regular file
	} else if (S_ISDIR(status.st_mode)) {
	    return 2;	// Directory
	}
	return 3;	// Special file
    }
}

// NOTE: "name" need normalize.
char *get_basename(const char *path) {
    char *start = (char*)path;

    for (; *path != '\0'; path++) {
	if ((*path == '/' || *path == '\\') && *(path+1) != '\0') {
	    start = (char*)++path;
	}
    }
    return start;
}

char *ifp_strcasestr(const char *base, const char *target) {
    size_t len;

    if ( (len = strlen(target)) <= 0) return (char *)base;

    for (; *base != '\0'; base++) {
	if (strncasecmp(base, target, len) == 0) {
	    return (char *)base;
	}
    }

    return NULL;
}

void normalize_path(char *path, size_t size) {
    char *pos;

    /* /path/to//file/// -> /path/to/file/ */
    remove_double_slash(path, size);

    /* /path/to/file/ -> /path/to/file */
    for (pos = path; *pos != '\0'; pos++)
	;
    if (*(pos-1) == '/' && pos - 1 != path) {
	*(pos-1) = '\0';
    }

}

void remove_double_slash(char *path, size_t size) {
    int nocopy = 0;
    int i;
    char *pos;

    for (i = 0, pos = path; i < (int)(size-1) && *path != '\0'; i++) {
	if (*path == '/') {
	    if (nocopy == 0) {
		*pos++ = *path++;
		nocopy = 1;
	    } else {
		path++;
	    }
	} else {
	    *pos++ = *path++;
	    nocopy = 0;
	}
    }
    *pos = '\0';
}

void backslash2slash(char *dest, const char *src, size_t count) {
    for (; *src != '\0' && count > 0; count--) {
	if (*src == '\\') {
	    *dest++ = '/'; src++;
	} else {
	    *dest++ = *src++;
	}
    }
    *dest = '\0';
}
void slash2backslash(char *dest, const char *src, size_t count) {
    for (; *src != '\0' && count > 0; count--) {
	if (*src == '/') {
	    *dest++ = '\\'; src++;
	} else {
	    *dest++ = *src++;
	}
    }
    *dest = '\0';
}

int load_device(struct ifp_device * dev, int quiet) {
	int i = 0;
	char foo[80];
	usb_dev_handle *dh;
	struct usb_device *rawdev = NULL;

	dh = ifp_find_device();
	if (dh == NULL) {
		fprintf(stderr, "A suitable iRiver iFP device couldn't be found; "
		    "perhaps it's unplugged or turned off.\n");
		goto out_0;
	}

	rawdev = usb_device(dh);
	/* "must be called" written in the libusb documentation */
	if (usb_claim_interface(dh, rawdev->config->interface->altsetting->bInterfaceNumber))
	{
		fprintf(stderr, "Device is busy.  (I was unable to claim its interface.)\n");
		goto out_1;
	}

	i = ifp_init(dev, dh);
	if (i) {
		printf("Device isn't responding.. try jiggling the handle. (error %d)\n",i);

		goto out_2;
	}

	if (!quiet) {
		i = ifp_device_info(dev, foo, sizeof(foo));
		if (i) {
			printf("device info failed, i=%d.\n", i);
			goto out_3;
		}
		printf("Detected: %s\n", foo);
	}

	return i;

out_3:
	i = ifp_finalize(dev);
	if (i) {
		fprintf(stderr, "warning: finalize failed, i=%d\n",i);
	}

out_2:
	usb_release_interface(dh,
		rawdev->config->interface->altsetting->bInterfaceNumber);
out_1:
	i = ifp_release_device(dh);
        if (i) {
		fprintf(stderr, "warning: release_device failed, i=%d\n",i);
	}
out_0:
	return i?i:1;
}

int return_device(struct ifp_device * dev) {
	int i = 0;
	usb_dev_handle *dh = dev->device;
	struct usb_device *rawdev = NULL;

	i = ifp_finalize(dev);
	if (i) {
		fprintf(stderr, "warning: finalize failed, i=%d\n",i);
	}

	rawdev = usb_device(dh);
	usb_release_interface(dh,
		rawdev->config->interface->altsetting->bInterfaceNumber);
	i = ifp_release_device(dh);
        if (i) {
		fprintf(stderr, "warning: release_device failed, i=%d\n",i);
	}
	return i;
}

int main(int argc, char **argv)
{
    struct ifp_device ifpdev;
    struct ifp_device *dev = &ifpdev;
    char ** cur_arg = argv + 2;
    int cur_cnt = argc - 2;
    int i = 0;

    FILE *fp = NULL;
    int is_mc;
    int retval = 0;
    //char remotepath[IFP_MAXPATHLEN];
    char buf[255];
    char * cmd = argv[1];
    char * mc_arg = NULL;


    /* check for mc (run from mc's extfs directory ? */
    if (strstr(argv[0],"/mc/extfs/")) {
	is_mc = IS_MC;
	mc_arg = *cur_arg;
	cur_arg++;
	cur_cnt--;
    } else {
	is_mc = IS_IFP;
    }

    /* no command */
    if (argc < 2) {
	help(argv[0]);
	return 0;
    }
	    
    usb_init();

    retval = load_device(dev, is_mc == IS_MC);
    ifp_err_expect(retval, retval == 1, out_0, "unable to init hardware");

    /* parse command */
    if (strcmp(cmd, "ls") == 0) {
	if (argc < 3) {
	    retval = print_directory(dev, "/");
	} else {
	    retval = print_directory(dev, argv[2]);
	}

    } else if (strcmp(cmd, "df") == 0) {
	printf("Capacity: %d bytes\n", ifp_capacity(dev));
	printf("Free:     %d bytes\n", ifp_freespace(dev));

    } else if (strcmp(cmd, "put") == 0) {
	retval = ifp_put(dev, cur_cnt, cur_arg);

    } else if (strcmp(cmd, "upload") == 0) {
	retval = upload(dev, cur_cnt, cur_arg);

    } else if (strcmp(cmd, "get") == 0) {
	retval = ifp_get(dev, cur_cnt, cur_arg);

    } else if (strcmp(cmd, "download") == 0) {
	retval = download(dev, cur_cnt, cur_arg);

    } else if (strcmp(cmd, "rm") == 0) {
	retval = ifp_rm(dev, cur_cnt, cur_arg);

    } else if (strcmp(cmd, "rmdir") == 0) {
	retval = ifp_delete_dir(dev, cur_cnt, cur_arg);

    } else if (strcmp(cmd, "mkdir") == 0) {
	retval = ifp_make_dir(dev, cur_cnt, cur_arg);

    } else if (strcmp(cmd, "battery") == 0) {
	printf("Battery status (really not sure!): %d\n",
		ifp_battery(dev));

    } else if (strcmp(cmd, "typestring") == 0) {
        //retval = ifp_device_info(dev, buf, sizeof(buf));
	retval = ifp_model(dev, buf, sizeof(buf));
	printf("iFP type string: %s\n", buf);

    } else if (strcmp(cmd, "firmversion") == 0) {
	int f = ifp_firmware_version(dev);
	printf("Firmware version (really not sure!): %2x.%02x\n",
		f / 0x100, f % 0x100);
    } else if (strcmp(cmd, "firmupdate") == 0) {
	//after this command, we are going to exit immediately.. which is good
	retval = firmware_update(dev, cur_cnt, cur_arg);
    } else if (strcmp(cmd, "format") == 0) {
	retval = format_media(dev);
    } else if (strcmp(cmd, "getpreset") == 0) {
	retval = tuner_show(dev);
    } else if (strcmp(cmd, "setpreset") == 0) {
	retval = tuner_set(dev, cur_cnt, cur_arg);
    // Functions for mc support
    } else if (strcmp(cmd, "copyin") == 0) {
	if (cur_cnt < 2) {
	    retval = -1;
	} else {
	    // suppress to upload file with name of MC_INFO
	    if (!strstr(cur_arg[0], MC_INFO)) { 
		if ( (fp = fopen(cur_arg[1], "r")) == NULL) {
		    perror(cur_arg[1]);
		    retval = -1;
		} else {
		    retval = ifp_write_file_progress(dev, fp, 0, cur_arg[0], NULL, NULL);
		    fclose(fp);
		}
		if (retval <= -2) fprintf(stderr, "Not enough space on iFP device.\n");
	    }
	}

    } else if (strcmp(cmd, "copyout") == 0) {
	if (cur_cnt < 2) {
	    retval = -1;
	} else {
	    if (strcmp(cur_arg[0], MC_INFO) == 0) {
		if ( (fp = fopen(cur_arg[1], "w")) == NULL) {
		    perror(cur_arg[1]);
		    retval = -1;
		} else {
		    info(dev, fp, 1);
		    fclose(fp);
		}

	    } else {
		// file to file only
		if ( (fp = fopen(cur_arg[1], "w")) == NULL) {
		    perror(cur_arg[1]);
		    retval = -1;
		} else {
		    retval = ifp_read_file_progress(dev, fp, cur_arg[0], NULL, NULL);
		    fclose(fp);
		}
	    }
	}

    } else if (strcmp(cmd, "list") == 0) {
	// faked ls -l
	/* faked file containing generated information about iFP */
	printf("-r-xr-xr-x  1 root  root        0 Jan  1 1980 " MC_INFO "\n");
	/* ls listing (all files and directories) */
	retval = list_all_files(dev);

    } else if (strcmp(cmd, "run") == 0) {
	/*if (argc >= 4)  */
	if (cur_cnt >= 1) {
	    if (strstr(cur_arg[0], MC_INFO)) {
		info(dev, stdout, 0);
		printf("### Press enter ###\n");
		getchar();
	    }
	}

    } else {
	help(argv[0]);
    }

    if (retval != 0 && retval != 1) {
	    printf("%s\n", ifp_error_message(retval));
	
    }

//out_1:
	i = return_device(dev);
        if (i) { ifp_err_i(i, "warning: return_device"); }
out_0:
	return retval ? 1 : 0;


}

void help(char *prog) {
    printf("usage: %s command params\n",prog);
    printf("commands:\n");
    printf("    ls [directory]\n");
    printf("    df\n");
    printf("    upload localfile ifptarget\n");
    printf("    upload localdir  ifpdir\n");
    printf("    put localfile\n");
    printf("    put localdir\n");
    printf("    download ifpfile localtarget\n");
    printf("    download ifpdir  localdir\n");
    printf("    get ifpfile\n");
    printf("    get ifpdir\n");
    printf("    rm [-r] file\n");
    printf("    rmdir dir\n");
    printf("    mkdir dir\n");
    printf("    battery\n");
    printf("    typestring\n");
    printf("    firmversion\n");
    printf("    format\n");
    printf("    firmupdate /path/to/FIRMWARE.HEX\n");
    printf("    getpreset\n");
    printf("    setpreset <slot number> <name> <frequency>\n");
    printf("\n");
    printf("Note: This program cannot work with UMS firmware.\n");
    printf("\n");
}

