## Usage

Expands numbers and letters, optionally using a `step` as the last argument. _(Numbers may be defined as JavaScript numbers or strings)_.

```js
var fill = require('{%= name %}');

console.log(fill('a', 'e'));
//=> ['a', 'b', 'c', 'd', 'e']

console.log(fill(0, 25, 5));
//=> [ 0, 5, 10, 15, 20, 25 ]

console.log(fill('a', 'e', {toRegex: true}));
//=> '[a-e]'

console.log(fill('a', 'z', 3, {toRegex: true}));
//=> 'a|d|g|j|m|p|s|v|y'

console.log(fill('1', '100', {toRegex: true}));
//=> '[1-9]|[1-9][0-9]|100'
```

Create regex-compatible ranges (returns a string, which can be used however you need to create a regex):

```js
console.log(fill('a', 'e', {toRegex: true}));
//=> '[a-e]'

console.log(fill('a', 'z', 3, {toRegex: true}));
//=> 'a|d|g|j|m|p|s|v|y'

console.log(fill('1', '100', {toRegex: true}));
//=> '[1-9]|[1-9][0-9]|100'
```

**Params**

```js
fill(start, stop, step, options, fn);
```

 - `start`: **{String|Number}** the number or letter to start with
 - `end`: **{String|Number}** the number or letter to end with
 - `step`: **{String|Number}** optionally pass the step to use. works for letters or numbers.
 - `options`: **{Object}**:
    + `toRegex`: return a regex-compatible string (still returned as an array for consistency)
    + `step`: pass the step on the options as an alternative to passing it as an argument
    + `strict`: `undefined` by default, set to true to throw errors on invalid ranges. 
 - `fn`: **{Function}** optionally [pass a function](#custom-function) to modify each character. This can also be defined as `options.transform`


**Examples**

```js
fill(1, 3)
//=> ['1', '2', '3']

fill('1', '3')
//=> ['1', '2', '3']

fill('0', '-5')
//=> [ '0', '-1', '-2', '-3', '-4', '-5' ]

fill(-9, 9, 3)
//=> [ '-9', '-6', '-3', '0', '3', '6', '9' ])

fill('-1', '-10', '-2')
//=> [ '-1', '-3', '-5', '-7', '-9' ]

fill('1', '10', '2')
//=> [ '1', '3', '5', '7', '9' ]

fill('a', 'e')
//=> ['a', 'b', 'c', 'd', 'e']

fill('a', 'e', 2)
//=> ['a', 'c', 'e']

fill('A', 'E', 2)
//=> ['A', 'C', 'E']
```

### Invalid ranges

When an invalid range is passed, `null` is returned. 

```js
fill('1.1', '2');   // decimals not supported in ranges
//=> null

fill('a', '2');     // unmatched values
//=> null

fill(1, 10, 'foo'); // invalid step
//=> null
```

If you want errors to be throw, set `options.strict` to true.


### Custom function

Optionally pass a custom function as last argument or on `options.transform`.

```js
// increase padding by two
var arr = fill('01', '05', function(val, a, b, step, idx, arr, options) {
  return repeat('0', (options.maxLength + 2) - val.length) + val;
});

console.log(arr);
//=> ['0001', '0002', '0003', '0004', '0005']
```
