/*	ByteBuffer_OutputStream

PIRL CVS ID: ByteBuffer_OutputStream.java,v 1.3 2012/04/16 06:18:24 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Utilities;

import	java.nio.Buffer;
import	java.nio.ByteBuffer;
import	java.io.OutputStream;
import	java.io.EOFException;


/**	A <i>ByteBuffer_OutputStream</i> presents a ByteBuffer as an OutputStream.
<p>
	This class is thread safe: Methods that might modify the backing
	ByteBuffer are synchronized on the ByteBuffer.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	1.3
	@see		ByteBuffer
	@see		OutputStream
*/
public class ByteBuffer_OutputStream
	extends OutputStream
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.ByteBuffer_OutputStream (1.3 2012/04/16 06:18:24)";

private ByteBuffer
	Destination;


private static String
	NL;
static
{
if ((NL = System.getProperty ("line.separator")) == null)
	 NL = "\n";
}

/*==============================================================================
	Constructors
*/
/**	Construct a ByteBuffer_OutputStream on a ByteBuffer.
<p>
	@param	destination	The ByteBuffer to be used as the destination of
		output data.
	@throws	NullPointerException	If the destination is null.
*/
public ByteBuffer_OutputStream
	(
	ByteBuffer	destination
	)
{
if ((Destination = destination) == null)
	throw new NullPointerException (ID + NL
		+ "A buffer must be provided.");
}

/**	A ByteBuffer_OutputStream must have a ByteBuffer.
*/
private ByteBuffer_OutputStream ()
{}

/*==============================================================================
	Accessors
*/
/**	Get the ByteBuffer backing this ByteBuffer_OutputStream.
<p>
	@return	The ByteBuffer backing this ByteBuffer_OutputStream.
*/
public ByteBuffer Destination ()
{return Destination;}

/*==============================================================================
	OutputStream
*/
/**	Get the amount of storage space remaining in the destination.
<p>
	@return The number of bytes remaining in the {@link #Destination()
		destination}. This is the difference between the buffer's
		current output {@link Buffer#position() position} and the output
		{@link Buffer#limit() limit}. Note that the limit may be less
		than the buffer {@link Buffer#capacity() capacity}.
*/
public int available ()
{return Destination.remaining ();}

/**	Put the datum at the next position in the destination.
<p>
	The operation is synchronized on the destination buffer.
<p>
	@throws	EOFExcetption	If the destination is full; i.e. the output
		position is at its limit.
	@see	#available()
*/
public void write 
	(
	int	datum
	)
	throws EOFException
{
synchronized (Destination)
	{
	if (Destination.hasRemaining ())
		{
		Destination.put ((byte)datum);
		return;
		}
	}
throw new EOFException (ID + NL
	+ "Can't write to a full buffer.");
}

/**	Put the contents of a byte array into the destination.
<p>
	This method simply uses
	<code>write (byte_array, 0, byte_array.length)</code>.
<p>
	@param	byte_array	A byte[] array containing the data to be stored
		in the destination buffer.
	@throws	NullPointerException	If the byte array is null.
	@throws	EOFExcetption	If the destination is full; i.e. the output
		position is at its limit.
	@see	#write(byte[], int, int)
	@see	#available()
*/
public void write
	(
	byte[]	byte_array
	)
	throws EOFException
{write (byte_array, 0, byte_array.length);}

/**	Put some number of bytes from a byte array into the destination.
<p>
	If the amount to write is less than or equal to zero nothing is done.
	The amount specified to write must be no greater than the number of
	bytes in the array after the offset nor the space remaining in the
	destination buffer.
<p>
	The operation is synchronized on the destination buffer.
<p>
	@param	byte_array	A byte[] array containing data to be transferred.
	@param	offset	The index of the array containing the first byte to be
		transferred.
	@param	amount	The maximum number of bytes to transfer from the array
		to the {@link #Destination() destination}.
	@throws	NullPointerException	If the byte array is null.
	@throws	IndexOutOfBoundsException	If the offset is negative or
		the offset plus the amount is greater than the remaining bytes
		in the array.
	@throws	EOFExcetption	If the space {@link #available() available}
		in the destination is less than the amount of data specified to
		transfer.
*/
public void write
	(
	byte[]	byte_array,
	int		offset,
	int		amount
	)
	throws EOFException
{
synchronized (Destination)
	{
	if (byte_array == null)
		throw new NullPointerException (ID + NL
			+ "Can't write from a null array.");
	if (amount <= 0)
		return;
	if (offset < 0 ||
		(offset + amount) > byte_array.length)
		throw new IndexOutOfBoundsException (ID + NL
			+ "Can't write from the array of length " + byte_array.length
				+ " starting at offset " + offset
				+ " the specified " + 
				+ amount + " byte" + ((amount == 1) ? "" : "s") + '.');
	if (amount > Destination.remaining ())
		throw new EOFException (ID + NL
			+ "Can't write from the array of length " + byte_array.length
				+ " starting at offset " + offset
				+ " the specified " + 
				+ amount + " byte" + ((amount == 1) ? "" : "s") + NL
			+ "because only " + Destination.remaining ()
				+ " byte" + ((Destination.remaining () == 1) ? "" : "s")
				+ " are available in the buffer.");
	Destination.put (byte_array, offset, amount);
	}
}


}
