# SPDX-FileCopyrightText: Christian Amsüss and the aiocoap contributors
#
# SPDX-License-Identifier: MIT

import warnings

from ..message import UndecidedRemote
from .. import interfaces

from ..util import hostportsplit


class ProxyForwarder(interfaces.RequestProvider):
    """Object that behaves like a Context but only provides the request
    function and forwards all messages to a proxy.

    This is not a proxy itself, it is just the interface for an external
    one."""

    def __init__(self, proxy_address, context):
        if "://" not in proxy_address:
            warnings.warn(
                "Proxy addresses without scheme are deprecated, "
                "please specify like `coap://host`, `coap+tcp://ip:port` "
                "etc.",
                DeprecationWarning,
            )
            proxy_address = "coap://" + proxy_address

        self.proxy_address = UndecidedRemote.from_pathless_uri(proxy_address)
        self.context = context

    proxy = property(lambda self: self._proxy)

    def request(self, message, **kwargs):
        if not isinstance(message.remote, UndecidedRemote):
            raise ValueError(
                "Message already has a configured "
                "remote, set .opt.uri_{host,port} instead of remote"
            )
        host, port = hostportsplit(message.remote.hostinfo)
        message.opt.uri_port = port
        message.opt.uri_host = host
        message.opt.proxy_scheme = message.remote.scheme
        message.remote = self.proxy_address

        return self.context.request(message)
