#!/usr/bin/env python3
from distutils.core import setup
from distutils.command.build_py import build_py
from distutils.command.clean import clean
from distutils.cmd import Command
import distutils.log
from cyclograph.version import VERSION
import glob
import os
import subprocess

files = ["openlayers.html",
         "map.html",
         "google.css",
         "draw_on_map.js"]

icons = glob.glob('pixmaps/32x32/*.png') + glob.glob('pixmaps/*svg')

class pyrcc(Command):
    description = "Run pyrcc to generate pyqt resource file"
    user_options = []
    def initialize_options(self):
        pass
    def finalize_options(self):
        pass
    def run(self):
        command = ['/usr/bin/pyrcc5',
                   '-o', 'cyclograph/qt/cyclograph_rc.py',
                   'qt_ui/cyclograph.qrc']
        self.announce("Running: %s" % " ".join(command), level=distutils.log.INFO)
        #We set the seed to 0 as environmental variable to obtain
        #a reproducible build
        subprocess.check_call(command, env={'QT_HASH_SEED':'0'})

class genhtml(Command):
    description = "Generate html user manual from asciidoc file"
    user_options = []
    def initialize_options(self):
        pass
    def finalize_options(self):
        pass
    def run(self):
        command = ['asciidoc', 
                   '-b', 'html5', 
                   '--section-numbers', 
                   '-a', 'footer-style=none', 
                   'doc/manual.txt']
        subprocess.check_call(command)

class pyuic(Command):
    description = "Generate python from designer ui files"
    user_options = []
    def initialize_options(self):
        pass
    def finalize_options(self):
        pass
    def run(self):
        command = ['./build.py']
        #The script is designed to be run from the qt_ui directory
        subprocess.check_call(command, cwd='qt_ui')

class msgfmt(Command):
    description = "Run gettext to generate .mo files"
    user_options = []
    def initialize_options(self):
        pass
    def finalize_options(self):
        pass
    def run(self):
        for lang in ['en', 'it']:
            filename = 'po/%s/LC_MESSAGES/cyclograph' % lang
            fnamein = filename + '.po'
            fnameout = filename + '.mo'
            subprocess.check_call(['msgfmt', fnamein, '-o', fnameout])

class generate_all(Command):
    description = "Run all the file generators"
    user_options = []
    def initialize_options(self):
        pass
    def finalize_options(self):
        pass
    def run(self):
        self.run_command("pyrcc")
        self.run_command("pyuic")
        self.run_command("html")
        self.run_command("msgfmt")

class my_build_py(build_py):
    def run(self):
        nogeneratepath = os.path.join(os.path.dirname(__file__), 'no-generate.stamp')
        if not os.path.exists(nogeneratepath):
            self.run_command('generate')
        build_py.run(self)

class my_clean(clean):
    def run(self):
        files = ['cyclograph/qt/cyclograph_rc.py',
                 'cyclograph/qt/ui_qt.py',
                 'doc/manual.html']
        files += glob.glob('qt_ui/ui_*.py')
        files += ['po/en/LC_MESSAGES/cyclograph.mo',
                  'po/it/LC_MESSAGES/cyclograph.mo']
        for f in files:
            try:
                os.remove(f)
            except OSError as e:
                pass #print(e)
        #run standard clean
        clean.run(self)

setup(name = "cyclograph",
    version = VERSION,
    description = "route altimetry plotting application",
    author = "Federico Brega, Pierluigi Villani",
    author_email = "charon66@users.sourceforge.net",
    url = "http://cyclograph.sourceforge.net",
    license = "GPL v3 or later",
    packages = ['cyclograph',
                'cyclograph.qt',
                'cyclograph.gtk3'],
    cmdclass = {'build_py': my_build_py,
                'clean': my_clean,
                'generate': generate_all,
                'pyrcc': pyrcc,
                'pyuic': pyuic,
                'msgfmt': msgfmt,
                'html': genhtml},
    package_data = {'cyclograph' : files },
    scripts = ["cyclograph/cyclograph"],
    data_files = [('share/icons/hicolor/scalable/apps', ['pixmaps/cyclograph.svg']),
        ('share/icons/hicolor/16x16/apps', ['pixmaps/16x16/cyclograph.png']),
        ('share/icons/hicolor/22x22/apps', ['pixmaps/22x22/cyclograph.png']),
        ('share/icons/hicolor/24x24/apps', ['pixmaps/24x24/cyclograph.png']),
        ('share/icons/hicolor/32x32/apps', ['pixmaps/32x32/cyclograph.png']),
        ('share/icons/hicolor/48x48/apps', ['pixmaps/48x48/cyclograph.png']),
        ('share/icons/hicolor/256x256/apps', ['pixmaps/256x256/cyclograph.png']),
        ('share/cyclograph/icons', icons),
        ('share/locale/en/LC_MESSAGES',
            ['po/en/LC_MESSAGES/cyclograph.mo']),
        ('share/locale/it/LC_MESSAGES',
            ['po/it/LC_MESSAGES/cyclograph.mo']),
        ('share/applications',
            ['cyclograph-qt.desktop',
             'cyclograph-gtk3.desktop']),
        ],
    long_description = \
"""CycloGraph is an application for plotting the elevation profile of routes.
Its main value is in the graphical visualization of the difficulty of a
road, in term of slope, difference in height, etc. Plots like these are
often used in cycling competitions, but are also useful in other sports,
such as hiking or running."""
) 
